# -*- coding: utf-8 -*-
from collections import namedtuple
import json
import logging
import time

from passport.backend.core.builders.frodo.utils import (
    frodo_bool,
    get_hint_metadata,
    get_password_metadata,
    get_phone_number_hash,
)
from passport.backend.core.conf import settings
from passport.backend.core.cookies.cookie_l import (
    CookieL,
    CookieLUnpackError,
)
from passport.backend.core.cookies.cookie_my import (
    CookieMy,
    CookieMyBaseError,
)
from passport.backend.core.types.login.login import normalize_login
from passport.backend.core.types.phone_number.phone_number import PhoneNumber


log = logging.getLogger('passport.frodo')


# Документация по передаваемым во ФРОДО параметрам:
# http://wiki.yandex-team.ru/passport/python/frodo#tekushhievyzovyvofrodooboronu
class FrodoInfo(namedtuple(
    'FrodoInfoBase', 'uid login firstname lastname email password '
    'password_ex hint_question_id hint_question_metadata hint_question_metadata_ex '
    'hint_answer_metadata hint_answer_metadata_ex service phone_number phone_number_hash '
    'yandexuid yandex_gid useragent host server_time social_provider ip_from fuid val_key language '
    'country action suggest_login_count captcha_generate_count step1time step2time '
    'phone_confirmation_first_sms_send_at phone_confirmation_first_code_checked '
    'phone_validation_changes phone_validation_error '
    'phone_confirmation_sms_count phone_confirms_count consumer origin '
    'is_suggested_login language_sys locale geo_coarse hardware_id os_id application '
    'cell_provider hardware_model clid ip app_uuid password_quality old_password_quality image_captcha_type '
    'suggest_login_length suggest_login_last_call '
    'phone_confirmation_last_send_at phone_confirmation_last_checked phone_confirmation_send_ip_limit_reached '
    'phone_confirmation_send_count_limit_reached phone_confirmation_confirms_count_limit_reached '
    'voice_captcha_type sanitize_phone_first_call sanitize_phone_last_call suggest_login_first_call '
    'login_validation_first_call login_validation_last_call password_validation_first_call '
    'password_validation_last_call captcha_checked_at track_created '
    'captcha_generated_at password_validation_count captcha_check_count '
    'login_validation_count sanitize_phone_count '
    'account_country account_language account_timezone accept_language is_ssl is_ssl_session_cookie_valid '
    'has_cookie_l has_cookie_yandex_login has_cookie_my has_cookie_ys has_cookie_yp '
    'cookie_my_block_count cookie_my_language cookie_l_login cookie_l_uid cookie_l_timestamp '
    'session_age session_ip session_create_timestamp account_karma page_loading_info '
    'phone_bindings_count'
)):

    def __new__(cls, **kwargs):
        for i in cls._fields:
            if i not in kwargs:
                kwargs[i] = ''
        return super(FrodoInfo, cls).__new__(cls, **kwargs)

    @classmethod
    def create(cls, env, query_params, track=None):
        user_agent = env.user_agent or ''
        service_from_track = track.service if track else None
        service_from_query = query_params.get('service')
        service = service_from_track or service_from_query or ''

        val_key = 0
        # TODO: решить что делать со 2ым битом
        # саджестор вызван не методом get и с неправильным заголовком для ajax запроса.
        # В нынешней версии апи саджестор нельзя будет вызвать не с тем методом
        if user_agent == '':
            val_key |= 4

        password, password_ex = get_password_metadata(query_params.get('password'))

        hint_question_id = query_params.get('hint_question_id', '')
        # FIXME: Найдено магическое число!
        hint = query_params.get('hint_question', '') if hint_question_id == 99 else ''
        hint_question_metadata, hint_question_metadata_ex = get_hint_metadata(hint)

        hint_answer_metadata, hint_answer_metadata_ex = get_hint_metadata(query_params.get('hint_answer', ''))

        login = query_params.get('login', '')
        uid_from_query = query_params.get('uid')
        uid_from_track = track.uid if track else ''
        uid = uid_from_track or uid_from_query or ''
        suggested_logins = track.suggested_logins.get() if track else []

        phone_number = cls.get_phone_number(env, query_params, track)
        if phone_number is not None:
            phone_number_hash = get_phone_number_hash(phone_number.e164)
            phone_number = phone_number.masked_format_for_frodo
        else:
            phone_number_hash = phone_number = ''

        try:
            cookie_my = CookieMy(env.cookies.get('my'))
        except CookieMyBaseError:
            cookie_my = None

        cookie_l_info = {}
        try:
            cookie_l = env.cookies.get('L')
            if cookie_l:
                cookie_l_info = CookieL().unpack(cookie_l)
                cookie_l_info['timestamp'] = cookie_l.split('.')[1]
        except CookieLUnpackError:
            pass

        page_loading_info = ''
        if track and track.page_loading_info:
            page_loading_info = track.page_loading_info
        elif track and (track.check_css_load or track.check_js_load):
            data = {
                key: int(value) for key, value in (('checkcssload', track.check_css_load), ('checkjsload', track.check_js_load)) if value
            }

            page_loading_info = json.dumps(data)

        if track and track.is_fake_client:
            ip = settings.FRODO_DUMMY_IP
            host = ''
            user_agent = ''
        else:
            ip = env.user_ip or ''
            host = env.host or ''

        return cls(
            uid=uid,
            login=normalize_login(login),
            firstname=query_params.get('firstname', ''),
            lastname=query_params.get('lastname', ''),
            email=query_params.get('email', ''),
            service=service,
            account_karma=query_params.get('account_karma', ''),

            password=password,
            password_ex=password_ex,
            password_quality=query_params.get('quality', ''),
            old_password_quality=query_params.get('old_password_quality', ''),

            hint_question_id=hint_question_id,
            hint_question_metadata=hint_question_metadata,
            hint_question_metadata_ex=hint_question_metadata_ex,

            hint_answer_metadata=hint_answer_metadata,
            hint_answer_metadata_ex=hint_answer_metadata_ex,

            phone_number=phone_number,
            phone_number_hash=phone_number_hash,

            yandexuid=env.cookies.get('yandexuid', ''),
            yandex_gid=env.cookies.get('yandex_gid', ''),
            fuid=env.cookies.get('fuid01', env.cookies.get('fuid00', '')),

            useragent=user_agent,
            host=host,

            ip_from=ip,

            server_time=time.time(),

            social_provider=query_params.get('social_provider', ''),

            val_key='{0:010b}'.format(val_key),

            language=query_params.get('language', ''),
            country=query_params.get('country', ''),
            action=query_params.get('action', ''),

            # TODO: Eсли mode=register, то брать данные из трека в случае многоходовой регистрации
            step1time=2000,
            step2time=2000,

            suggest_login_count=track.suggest_login_count.get() if track else '',
            suggest_login_length=(len(suggested_logins)
                                  if track and track.suggest_login_count.get() is not None
                                  else ''),
            captcha_generate_count=track.captcha_generate_count.get() if track else '',
            image_captcha_type=track.image_captcha_type if track else '',
            voice_captcha_type=track.voice_captcha_type if track else '',
            phone_confirmation_first_sms_send_at=track.phone_confirmation_first_send_at if track else '',
            phone_confirmation_first_code_checked=track.phone_confirmation_first_checked if track else '',
            phone_validation_changes=frodo_bool(track.sanitize_phone_changed_phone) if track else '',
            phone_validation_error=frodo_bool(track.sanitize_phone_error) if track else '',
            phone_confirmation_sms_count=track.phone_confirmation_sms_count.get() if track else '',
            phone_confirms_count=track.phone_confirmation_confirms_count.get() if track else '',
            phone_bindings_count=query_params.get('phone_bindings_count', ''),

            phone_confirmation_last_send_at=track.phone_confirmation_last_send_at if track else '',
            phone_confirmation_last_checked=track.phone_confirmation_last_checked if track else '',

            phone_confirmation_send_ip_limit_reached=frodo_bool(track.phone_confirmation_send_ip_limit_reached) if track else '',
            phone_confirmation_send_count_limit_reached=frodo_bool(track.phone_confirmation_send_count_limit_reached) if track else '',
            phone_confirmation_confirms_count_limit_reached=frodo_bool(track.phone_confirmation_confirms_count_limit_reached) if track else '',

            consumer=query_params.get('consumer', ''),
            origin=track.origin if track else '',
            is_suggested_login=(frodo_bool(login in suggested_logins)
                                if (track and track.suggest_login_count.get() is not None)
                                else ''),

            language_sys=track.device_language_sys if track else '',
            locale=track.device_locale if track else '',
            geo_coarse=track.device_geo_coarse if track else '',
            hardware_id=track.device_hardware_id if track else '',
            os_id=track.device_os_id if track else '',
            application=track.device_application if track else '',
            cell_provider=track.device_cell_provider if track else '',
            hardware_model=track.device_hardware_model if track else '',
            clid=track.device_clid if track else '',
            app_uuid=track.device_app_uuid if track else '',
            ip=ip,

            sanitize_phone_count=track.sanitize_phone_count.get() if track else '',
            sanitize_phone_first_call=track.sanitize_phone_first_call if track else '',
            sanitize_phone_last_call=track.sanitize_phone_last_call if track else '',

            suggest_login_first_call=track.suggest_login_first_call if track else '',
            suggest_login_last_call=track.suggest_login_last_call if track else '',

            login_validation_count=track.login_validation_count.get() if track else '',
            login_validation_first_call=track.login_validation_first_call if track else '',
            login_validation_last_call=track.login_validation_last_call if track else '',

            password_validation_count=track.password_validation_count.get() if track else '',
            password_validation_first_call=track.password_validation_first_call if track else '',
            password_validation_last_call=track.password_validation_last_call if track else '',

            captcha_generated_at=track.captcha_generated_at if track else '',
            captcha_checked_at=track.captcha_checked_at if track else '',
            captcha_check_count=track.captcha_check_count.get() if track else '',
            track_created=track.created if track else '',

            has_cookie_l=frodo_bool('L' in env.cookies),
            has_cookie_yandex_login=frodo_bool('yandex_login' in env.cookies),
            has_cookie_my=frodo_bool('my' in env.cookies),
            has_cookie_ys=frodo_bool('ys' in env.cookies),
            has_cookie_yp=frodo_bool('yp' in env.cookies),

            cookie_my_block_count=len(cookie_my._cookie.keys()) if cookie_my is not None else '',
            cookie_my_language=cookie_my.language if cookie_my is not None else '',
            cookie_l_login=cookie_l_info.get('login', ''),
            cookie_l_uid=cookie_l_info.get('uid', ''),
            cookie_l_timestamp=cookie_l_info.get('timestamp', ''),

            session_age=track.old_session_age if track else '',
            session_ip=track.old_session_ip if track else '',
            session_create_timestamp=track.old_session_create_timestamp if track else '',

            account_country=query_params.get('account_country', '').lower(),
            account_language=query_params.get('account_language', ''),
            account_timezone=query_params.get('account_timezone', ''),
            accept_language=env.accept_language or '',
            is_ssl=frodo_bool(True),
            is_ssl_session_cookie_valid=frodo_bool(query_params.get('is_ssl_session_cookie_valid')) or '',
            page_loading_info=page_loading_info,
        )

    @classmethod
    def get_phone_number(cls, env, query_params, track=None):
        phone_number = query_params.get('phone_number')
        if phone_number is None and track and track.phone_confirmation_is_confirmed:
            # Берем подтвержденный телефон из трека, если телефон не указан явно
            phone_number = PhoneNumber.parse(track.phone_confirmation_phone_number)
        if phone_number is not None and not isinstance(phone_number, PhoneNumber):
            phone_number = PhoneNumber.parse(phone_number)
        return phone_number
