# -*- coding: utf-8 -*-

from collections import namedtuple
import logging

from passport.backend.core.builders.base.base import (
    BaseBuilder,
    forward_content,
    RequestInfo,
)
from passport.backend.core.builders.frodo.exceptions import (
    FrodoError,
    FrodoRequestError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import (
    GraphiteLogger,
    StatboxLogger,
)
from passport.backend.core.types.login.login import is_test_frodo_login
from passport.backend.core.xml.xml import (
    XML,
    XMLParser,
    XMLSyntaxError,
)
from passport.backend.utils.string import smart_unicode
import yenv


log = logging.getLogger('passport.frodo')

User = namedtuple('user', 'login karma is_pdd spam change_pass')


TEST_RESPONSE = '<spamlist>%s</spamlist>'
TEST_RESPONSES = {
    'frodo-spam': lambda x: TEST_RESPONSE % '<spam_user login="%s" weight="85" />' % x,
    'frodo-pdd-spam': lambda x: TEST_RESPONSE % '<spam_user login="%s" pdd="yes" weight="100" />' % x,
    'frodo-change-pass': lambda x: TEST_RESPONSE % '<change_pass login="%s" weight="85" />' % x,
}


class Frodo(BaseBuilder):
    base_error_class = FrodoError
    temporary_error_class = FrodoRequestError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, graphite_logger=None):
        timeout = timeout or settings.FRODO_TIMEOUT
        graphite_logger = graphite_logger or GraphiteLogger(service='frodo')
        statbox_logger = StatboxLogger(action='call_frodo', destination='frodo')
        super(Frodo, self).__init__(
            url=url or settings.FRODO_URL,
            timeout=timeout,
            retries=retries or settings.FRODO_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            statbox_logger=statbox_logger,
        )

    def confirm(self, params):
        self._request_with_retries(
            u'GET',
            self._build_request(u'loginrcpt', params),
            self._parse_confirm_response,
            http_error_handler=self._handle_http_error,
        )
        return True

    def _parse_confirm_response(self, raw_response):
        return raw_response

    def _build_request(self, method, params):
        url = self.url + method
        if params:
            params = [
                (key, smart_unicode(value)) for key, value in params.items()
            ]
        return RequestInfo(url, params, None)

    def _handle_http_error(self, raw_response):
        if raw_response.status_code != 200:
            raise FrodoRequestError(
                u'Bad HTTP status code: %d, response: %s' % (
                    raw_response.status_code,
                    trim_message(smart_unicode(raw_response.content, errors='ignore')),
                ),
            )

    def check(self, frodo_info):
        # Специальный код для тестирования совместимости с ФО
        if yenv.name == 'localhost' and yenv.type in ('development', 'testing'):
            prefix = is_test_frodo_login(frodo_info.login)
            if prefix:
                return list(self._parse_check_response(TEST_RESPONSES.get(prefix)(frodo_info.login)))

        response = self._request_with_retries(
            u'GET',
            self._build_request(u'check', self._build_check_params(frodo_info)),
            forward_content(self._parse_check_response),
            http_error_handler=self._handle_http_error,
        )
        return list(response)

    def _build_check_params(self, frodo_info):
        """
        Собираем параметры для запроса во Фродо.
        Подробное описание параметров и их форматов тут: http://wiki.yandex-team.ru/passport/python/frodo
        """

        params = dict(so_codes='utf8', captchareq='0', ip_prox='')

        params['uid'] = frodo_info.uid
        params['login'] = frodo_info.login
        params['iname'] = frodo_info.firstname
        params['fname'] = frodo_info.lastname
        params['email'] = frodo_info.email  # возможно не будем передавать
        params['from'] = frodo_info.service

        params['passwd'] = frodo_info.password
        params['passwdex'] = frodo_info.password_ex

        params['hintqid'] = frodo_info.hint_question_id
        params['hintq'] = frodo_info.hint_question_metadata
        params['hintqex'] = frodo_info.hint_question_metadata_ex
        params['hinta'] = frodo_info.hint_answer_metadata
        params['hintaex'] = frodo_info.hint_answer_metadata_ex

        params['phonenumber'] = frodo_info.phone_number
        params['v2_phonenumber_hash'] = frodo_info.phone_number_hash

        params['yandexuid'] = frodo_info.yandexuid
        params['v2_yandex_gid'] = frodo_info.yandex_gid
        params['fuid'] = frodo_info.fuid

        params['useragent'] = frodo_info.useragent
        params['host'] = frodo_info.host

        params['utime'] = int(frodo_info.server_time * 1000)
        params['time'] = int(frodo_info.server_time)

        params['social_provider'] = frodo_info.social_provider

        params['ip_from'] = frodo_info.ip_from

        params['valkey'] = frodo_info.val_key

        params['lcheck'] = frodo_info.suggest_login_count

        params['captchacount'] = frodo_info.captcha_generate_count

        params['step1time'] = frodo_info.step1time
        params['step2time'] = frodo_info.step2time

        params['action'] = frodo_info.action
        params['lang'] = frodo_info.language
        params['xcountry'] = frodo_info.country

        params['v2_phone_confirmation_first_sms_send_at'] = frodo_info.phone_confirmation_first_sms_send_at
        params['v2_phone_confirmation_first_code_checked'] = frodo_info.phone_confirmation_first_code_checked
        params['v2_phone_validation_changes'] = frodo_info.phone_validation_changes
        params['v2_phone_validation_error'] = frodo_info.phone_validation_error
        params['v2_phone_confirmation_sms_count'] = frodo_info.phone_confirmation_sms_count
        params['v2_phone_confirmation_confirms_count'] = frodo_info.phone_confirms_count

        params['v2_phone_confirmation_send_ip_limit_reached'] = frodo_info.phone_confirmation_send_ip_limit_reached
        params['v2_phone_confirmation_send_count_limit_reached'] = frodo_info.phone_confirmation_send_count_limit_reached
        params['v2_phone_confirmation_confirms_count_limit_reached'] = frodo_info.phone_confirmation_confirms_count_limit_reached
        params['v2_phone_bindings_count'] = frodo_info.phone_bindings_count

        params['consumer'] = frodo_info.consumer
        params['origin'] = frodo_info.origin
        params['is_suggested_login'] = frodo_info.is_suggested_login

        params['v2_language_sys'] = frodo_info.language_sys
        params['v2_locale'] = frodo_info.locale
        params['v2_geo_coarse'] = frodo_info.geo_coarse
        params['v2_hardware_id'] = frodo_info.hardware_id
        params['v2_os_id'] = frodo_info.os_id
        params['v2_application'] = frodo_info.application
        params['v2_cell_provider'] = frodo_info.cell_provider
        params['v2_hardware_model'] = frodo_info.hardware_model
        params['v2_clid'] = frodo_info.clid
        params['v2_ip'] = frodo_info.ip
        params['v2_app_uuid'] = frodo_info.app_uuid
        params['v2_password_quality'] = frodo_info.password_quality
        params['v2_old_password_quality'] = frodo_info.old_password_quality
        params['v2_image_captcha_type'] = frodo_info.image_captcha_type
        params['v2_voice_captcha_type'] = frodo_info.voice_captcha_type
        params['v2_suggest_login_length'] = frodo_info.suggest_login_length
        params['v2_phone_confirmation_last_sms_send_at'] = frodo_info.phone_confirmation_last_send_at
        params['v2_phone_confirmation_last_code_checked'] = frodo_info.phone_confirmation_last_checked
        params['v2_sanitize_phone_count'] = frodo_info.sanitize_phone_count
        params['v2_sanitize_phone_first_call'] = frodo_info.sanitize_phone_first_call
        params['v2_sanitize_phone_last_call'] = frodo_info.sanitize_phone_last_call
        params['v2_suggest_login_first_call'] = frodo_info.suggest_login_first_call
        params['v2_suggest_login_last_call'] = frodo_info.suggest_login_last_call
        params['v2_login_validation_count'] = frodo_info.login_validation_count
        params['v2_login_validation_first_call'] = frodo_info.login_validation_first_call
        params['v2_login_validation_last_call'] = frodo_info.login_validation_last_call
        params['v2_password_validation_count'] = frodo_info.password_validation_count
        params['v2_password_validation_first_call'] = frodo_info.password_validation_first_call
        params['v2_password_validation_last_call'] = frodo_info.password_validation_last_call
        params['v2_captcha_generated_at'] = frodo_info.captcha_generated_at
        params['v2_captcha_checked_at'] = frodo_info.captcha_checked_at
        params['v2_track_created'] = frodo_info.track_created
        params['v2_captcha_check_count'] = frodo_info.captcha_check_count

        params['v2_has_cookie_l'] = frodo_info.has_cookie_l
        params['v2_has_cookie_yandex_login'] = frodo_info.has_cookie_yandex_login
        params['v2_has_cookie_my'] = frodo_info.has_cookie_my
        params['v2_has_cookie_ys'] = frodo_info.has_cookie_ys
        params['v2_has_cookie_yp'] = frodo_info.has_cookie_yp

        params['v2_cookie_my_block_count'] = frodo_info.cookie_my_block_count
        params['v2_cookie_my_language'] = frodo_info.cookie_my_language
        params['v2_cookie_l_login'] = frodo_info.cookie_l_login
        params['v2_cookie_l_uid'] = frodo_info.cookie_l_uid
        params['v2_cookie_l_timestamp'] = frodo_info.cookie_l_timestamp

        params['v2_session_age'] = frodo_info.session_age
        params['v2_session_ip'] = frodo_info.session_ip
        params['v2_session_create_timestamp'] = frodo_info.session_create_timestamp

        params['v2_account_karma'] = frodo_info.account_karma
        params['v2_account_country'] = frodo_info.account_country
        params['v2_account_language'] = frodo_info.account_language
        params['v2_account_timezone'] = frodo_info.account_timezone
        params['v2_accept_language'] = frodo_info.accept_language
        params['v2_is_ssl'] = frodo_info.is_ssl
        params['v2_is_ssl_session_cookie_valid'] = frodo_info.is_ssl_session_cookie_valid

        params['v2_page_loading_info'] = frodo_info.page_loading_info

        for key in params:
            if params[key] is None:
                params[key] = ''
        return params

    def _parse_user(self, node):
        is_pdd = node.get('pdd') == 'yes'
        spam = node.tag == 'spam_user'
        change_pass = node.tag == 'change_pass'
        return User(
            node.get('login'),
            int(node.get('weight')),
            is_pdd,
            spam,
            change_pass,
        )

    def _parse_check_response(self, response):
        try:
            xml_response = XML(response, parser=XMLParser())

            if xml_response.tag != 'spamlist':
                log.error(
                    u'SO response has no "spamlist" tag. Response text: "%s".',
                    smart_unicode(trim_message(response), errors='ignore'),
                )
                raise FrodoError('No "spamlist" in Frodo response')

            for node in xml_response.iter('spam_user'):
                yield self._parse_user(node)

            for node in xml_response.iter('change_pass'):
                yield self._parse_user(node)

        except XMLSyntaxError:
            log.warning(u'Frodo returned invalid XML: %s', smart_unicode(trim_message(response), errors='ignore'))
            raise FrodoError('Invalid XML in Frodo response')


def get_frodo():
    return Frodo()  # pragma: no cover
