# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.protobuf_parser.protobuf_parser import ProtobufParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.exceptions import BaseCoreError
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.utils.string import smart_unicode
from six.moves.urllib.parse import (
    quote_plus,
    urlparse,
)
from yandex.maps.proto.common2 import response_pb2
from yandex.maps.proto.search.business_pb2 import GEO_OBJECT_METADATA as BUSINESS_METADATA
from yandex.maps.proto.search.business_rating_pb2 import GEO_OBJECT_METADATA as BUSINESS_RATING_METADATA
from yandex.maps.proto.search.geocoder_pb2 import GEO_OBJECT_METADATA as GEOCODER_METADATA
from yandex.maps.proto.search.kind_pb2 import Kind
from yandex.maps.proto.search.photos_2x_pb2 import GEO_OBJECT_METADATA as PHOTOS_METADATA


log = logging.getLogger('passport.geosearch')


class BaseGeoSearchApiError(BaseCoreError):
    pass


class GeoSearchApiTemporaryError(BaseGeoSearchApiError):
    """Временная ошибка апи - стоит поретраиться"""


class GeoSearchApiPermanentError(BaseGeoSearchApiError):
    """Пятисотка или иная непредвиденная ошибка"""


def geosearch_api_error_detector(response, raw_response):
    if raw_response.status_code != 200:
        raise GeoSearchApiPermanentError(
            u'Request failed with with response=%s code=%s' % (
                trim_message(repr(raw_response)),
                raw_response.status_code,
            ),
        )


def geosearch_api_response_processor(response):
    result = {}

    geo_object = response.reply.geo_object
    if geo_object:
        geo_object = geo_object[0]
    else:
        return result

    if geo_object.geometry:
        point = geo_object.geometry[0].point
    else:
        point = None
    if point and point.lon is not None and point.lat is not None:
        result['coordinates'] = [point.lon, point.lat]
    else:
        result['coordinates'] = []

    result.update({
        'name': smart_unicode(geo_object.name),
        'description': smart_unicode(geo_object.description),
    })
    for metadata in geo_object.metadata:
        if metadata.HasExtension(BUSINESS_RATING_METADATA):
            business_rating = metadata.Extensions[BUSINESS_RATING_METADATA]
            result.update({
                'score': round(business_rating.score or 0, 1),
                'ratings': business_rating.ratings,
            })
        elif metadata.HasExtension(PHOTOS_METADATA):
            photos = metadata.Extensions[PHOTOS_METADATA].photo
            if photos:
                result.update({
                    'photo_url_template': photos[0].url_template,
                })
        elif metadata.HasExtension(BUSINESS_METADATA):
            business = metadata.Extensions[BUSINESS_METADATA]
            result.update({
                'type': 'org',
                'address': smart_unicode(business.address.formatted_address),
                'categories': [smart_unicode(c.name) for c in business.category if c.name],
            })
        elif metadata.HasExtension(GEOCODER_METADATA):
            geo = metadata.Extensions[GEOCODER_METADATA]
            result.update({
                'type': 'geo',
                'address': smart_unicode(geo.address.formatted_address),
            })
            if geo.address.component:
                result.update({
                    'kind': Kind.Name(geo.address.component[-1].kind[0]).lower(),
                    'text': smart_unicode(geo.address.component[-1].name) or u'',
                })
    return result


class GeoSearchApi(BaseBuilder, ProtobufParserMixin):
    """
    Документация:
    * https://wiki.yandex-team.ru/maps/dev/core/geobookmarks/#geo-bookmarksinacloud
    * https://wiki.yandex-team.ru/maps/api/http/internal/search/guide/
    """
    base_error_class = BaseGeoSearchApiError
    temporary_error_class = GeoSearchApiTemporaryError
    parser_error_class = GeoSearchApiPermanentError

    ALLOWED_GEO_TYPES = {'pin', 'geo', 'org'}

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, graphite_logger=None, **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='geosearch_api')
        super(GeoSearchApi, self).__init__(
            url=url or settings.GEOSEARCH_API_URL,
            timeout=timeout or settings.GEOSEARCH_API_TIMEOUT,
            retries=retries or settings.GEOSEARCH_API_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias='geosearch_api',
            **kwargs
        )

    def get_info(self, uri, lang='ru', parse_protobuf=True):
        """
        Виды uri:
        * ymapsbm1://org?oid=1364764998
        * ymapsbm1://org?oid=146311570
        * ymapsbm1://pin?ll=37.921005%2C55.766916
        * ymapsbm1://geo?text=Ленинский%20проспект&ll=37.536733%2C55.682336&spn=0.432587%2C0.095074
        * ymapsbm1://geo?text=Чили, Сантьяго, Парке О\'Игхинс

        parse_protobuf может быть полезен при отладке в интерпретаторе, даёт возможность
        покрутить ответ сервера в руках.
        """
        geo_type = urlparse(uri).netloc
        if geo_type not in self.ALLOWED_GEO_TYPES:
            return {}

        params = {
            'mode': 'uri',
            'uri': quote_plus(uri),
            'lang': lang,
            'ms': 'pb',
            'type': 'biz,geo',
            'origin': 'maps-passport',
            'snippets': 'businessrating/1.x,photos/2.x',
        }
        return self._request_with_retries_simple(
            headers={
                'Accept': 'application/x-protobuf',
            },
            url_suffix='',
            method='GET',
            params=params,
            parser=self.build_protobuf_parser(response_pb2.Response()),
            error_detector=geosearch_api_error_detector,
            response_processor=geosearch_api_response_processor if parse_protobuf else lambda x: x,
        )


def get_geosearch_api():
    return GeoSearchApi()   # pragma: no cover
