# -*- coding: utf-8 -*-

# Описание ручек API на Вики https://wiki.yandex-team.ru/passport/historydb/api

import logging

from passport.backend.core.builders.base.base import (
    BaseBuilder,
    forward_content,
    RequestInfo,
)
from passport.backend.core.builders.historydb_api.exceptions import (
    BaseHistoryDBApiError,
    HistoryDBApiPermanentError,
    HistoryDBApiTemporaryError,
)
from passport.backend.core.builders.historydb_api.parsers import (
    parse_auths,
    parse_auths_aggregated,
    parse_auths_aggregated_runtime,
    parse_auths_failed,
    parse_historydb_api_events,
    parse_historydb_api_events_passwords,
    parse_historydb_api_events_restore,
    parse_last_letter,
    parse_lastauth,
    parse_lastauth_bulk,
    parse_mail_history,
    parse_push_history,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from six.moves.urllib.parse import urljoin


log = logging.getLogger('passport.historydb_api.build_requests')

# Сортировка в порядке возрастания timestamp (по умолчанию - в порядке убывания) - для API версии 2
ORDER_BY_TIMESTAMP_ASCENDING = 'asc'
TRUE_VALUE = 'true'
FALSE_VALUE = 'false'


def historydb_http_error_handler(raw_response):
    if raw_response.status_code == 200:
        return

    log.warning(
        u'Request failed with response=%s code=%s',
        trim_message(raw_response.content.decode('utf-8')),
        raw_response.status_code,
    )

    if raw_response.status_code >= 500:
        raise HistoryDBApiTemporaryError(
            'Server is down (code=%s)' % raw_response.status_code,
        )

    if raw_response.status_code == 403:
        raise HistoryDBApiPermanentError('Invalid auth')

    raise HistoryDBApiPermanentError(
        'Bad response status code: %s' % raw_response.status_code,
    )


class HistoryDBApi(BaseBuilder):
    """
    https://wiki.yandex-team.ru/passport/historydb/api/
    """
    base_error_class = BaseHistoryDBApiError
    temporary_error_class = HistoryDBApiTemporaryError

    def __init__(self, historydb_api=None, consumer=None, useragent=None,
                 timeout=None, retries=None, graphite_logger=None, use_tvm=None, **kwargs):
        if useragent is not None and timeout is not None:
            raise ValueError('Timeout does not make sense when useragent is defined')
        if use_tvm is None:
            use_tvm = settings.HISTORYDB_API_USE_TVM
        self.consumer = consumer or settings.HISTORYDB_API_CONSUMER
        timeout = timeout or settings.HISTORYDB_API_TIMEOUT
        graphite_logger = graphite_logger or GraphiteLogger(service='historydb')
        super(HistoryDBApi, self).__init__(
            url=historydb_api or settings.HISTORYDB_API_URL,
            timeout=timeout,
            retries=retries or settings.HISTORYDB_API_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias='historydb_api' if use_tvm else None,
            **kwargs
        )

    def _request_builder(self, path, get_args, post_args=None):
        if get_args is None:
            get_args = {}
        get_args['consumer'] = self.consumer
        return RequestInfo(urljoin(self.url, path), get_args, post_args)

    def events(self, uid, from_ts, to_ts, limit=None, name=None, ascending=False):
        return self._request_with_retries(
            'GET',
            self.build_events_request(uid, from_ts, to_ts, limit, name, ascending),
            forward_content(parse_historydb_api_events),
        )

    def events_restore(self, uid, from_ts, to_ts, ascending=False):
        return self._request_with_retries(
            'GET',
            self.build_events_restore_request(uid, from_ts, to_ts, ascending),
            forward_content(parse_historydb_api_events_restore),
        )

    def events_passwords(self, uid, password, from_ts, to_ts):
        data = dict(
            uid=uid,
            password=password,
            from_ts=from_ts,
            to_ts=to_ts,
        )
        return self._request_with_retries(
            'POST',
            self.build_events_passwords_request(data),
            forward_content(parse_historydb_api_events_passwords),
        )

    def auths(self, uid, from_ts, to_ts, limit=None, ascending=False,
              statuses=None, auth_types=None, client_names=None):
        return self._request_with_retries(
            'GET',
            self.build_auths(uid, from_ts, to_ts, limit, ascending, statuses, auth_types, client_names),
            forward_content(parse_auths),
        )

    def auths_aggregated_old(self, uid, limit=None, hours_limit=None, from_row=None, password_auths=None):
        return self._request_with_retries(
            'GET',
            self.build_auths_aggregated_old(uid, limit, hours_limit, from_row, password_auths),
            forward_content(parse_auths_aggregated),
        )

    def auths_aggregated(self, uid, limit=None, hours_limit=None, from_row=None, password_auths=None):
        return self._request_with_retries(
            'GET',
            self.build_auths_aggregated(uid, limit, hours_limit, from_row, password_auths),
            forward_content(parse_auths_aggregated),
        )

    def auths_aggregated_runtime(self, uid, from_ts, to_ts, limit=None):
        """
        Информация об успешных авторизациях, аггрегированная по дням и окружению (браузер, ОС, yandexuid, тип авторизации).
        """
        return self._request_with_retries(
            'GET',
            self.build_auths_aggregated_runtime_request(
                uid,
                from_ts,
                to_ts,
                limit,
            ),
            forward_content(parse_auths_aggregated_runtime),
        )

    def auths_failed(self, uid, from_ts, to_ts, limit=None, status=None):
        return self._request_with_retries(
            'GET',
            self.build_auths_failed_request(uid, from_ts, to_ts, limit=limit, status=status),
            forward_content(parse_auths_failed),
        )

    def lastauth(self, uid):
        """
        Получение последней аутентификации в паспорте для заданного uid.
        """
        request = self._request_builder('/2/lastauth/', {'uid': uid})
        return self._request_with_retries(
            'GET',
            request,
            forward_content(parse_lastauth),
        )

    def lastauth_bulk(self, uids):
        """
        Получение последней аутентификации в паспорте для заданного списка uid.
        """
        uids_string = ','.join(str(uid) for uid in sorted(uids))
        request = self._request_builder('/2/lastauth/bulk/', {'uids': uids_string})
        return self._request_with_retries(
            'GET',
            request,
            forward_content(parse_lastauth_bulk),
        )

    def mail_history(self, uid, from_ts, to_ts, limit=None, is_corp=False, operations=None, modules=None):
        """
        Получение истории почтовых событий
        """
        return self._request_with_retries(
            'GET',
            self.build_mail_history_request(
                uid,
                from_ts,
                to_ts,
                limit=limit,
                is_corp=is_corp,
                operations=operations,
                modules=modules,
            ),
            forward_content(parse_mail_history),
            http_error_handler=historydb_http_error_handler,
        )

    def last_letter(self, user_ticket, max_age):
        """
        Получение дат последних отправленных почтовых рассылок
        """
        return self._request_with_retries(
            'GET',
            self.build_last_letter_request(max_age=max_age),
            headers={
                'X-Ya-User-Ticket': user_ticket,
            },
            parser=forward_content(parse_last_letter),
            http_error_handler=historydb_http_error_handler,
        )

    def build_events_request(self, uid, from_ts, to_ts, limit=None, name=None, ascending=False):
        request = dict(
            uid=uid,
            from_ts=from_ts,
            to_ts=to_ts,
        )
        if limit is not None:
            request['limit'] = limit
        if name is not None:
            request['name'] = name
        if ascending:
            request['order_by'] = ORDER_BY_TIMESTAMP_ASCENDING
        return self._request_builder('/2/events/', request)

    def build_events_restore_request(self, uid, from_ts, to_ts, ascending=False):
        request = dict(
            uid=uid,
            from_ts=from_ts,
            to_ts=to_ts,
        )
        if ascending:
            request['order_by'] = ORDER_BY_TIMESTAMP_ASCENDING
        return self._request_builder('/2/events/restore/', request)

    def build_events_passwords_request(self, data):
        return self._request_builder('/2/events/passwords/', None, post_args=data)

    def build_auths(self, uid, from_ts, to_ts, limit=None, ascending=False,
                    statuses=None, auth_types=None, client_names=None):
        request = dict(
            uid=uid,
            from_ts=from_ts,
            to_ts=to_ts,
        )
        if limit is not None:
            request['limit'] = limit

        if ascending:
            request['order_by'] = ORDER_BY_TIMESTAMP_ASCENDING

        if statuses is not None:
            request['status'] = ','.join(statuses)

        if auth_types is not None:
            request['type'] = ','.join(auth_types)

        if client_names is not None:
            request['client_name'] = ','.join(client_names)

        return self._request_builder('/2/auths/', request)

    def build_auths_aggregated_old(self, uid, limit=None, hours_limit=None, from_row=None, password_auths=None):
        request = dict(
            uid=uid,
        )

        if limit is not None:
            request['limit'] = limit

        if hours_limit is not None:
            request['hours_limit'] = hours_limit

        if from_row is not None:
            request['from'] = from_row

        if password_auths is not None:
            request['password_auths'] = TRUE_VALUE if password_auths else FALSE_VALUE

        return self._request_builder('/2/auths/aggregated/', request)

    def build_auths_aggregated(self, uid, limit=None, hours_limit=None, from_row=None, password_auths=None):
        request = dict(
            uid=uid,
        )

        if limit is not None:
            request['limit'] = limit

        if hours_limit is not None:
            request['hours_limit'] = hours_limit

        if from_row is not None:
            request['from'] = from_row

        if password_auths is not None:
            request['password_auths'] = TRUE_VALUE if password_auths else FALSE_VALUE

        return self._request_builder('/3/auths/aggregated/', request)

    def build_auths_aggregated_runtime_request(self, uid, from_ts, to_ts, limit=None):
        request = dict(
            uid=uid,
            from_ts=from_ts,
            to_ts=to_ts,
        )
        if limit is not None:
            request['limit'] = limit
        return self._request_builder('/2/auths/aggregated/runtime/', request)

    def build_auths_failed_request(self, uid, from_ts, to_ts, limit=None, status=None):
        request = dict(
            uid=uid,
            from_ts=from_ts,
            to_ts=to_ts,
        )
        if limit is not None:
            request['limit'] = limit
        if status is not None:
            request['status'] = status
        return self._request_builder('/2/auths/failed/', request)

    def build_mail_history_request(self, uid, from_ts, to_ts, limit=None, is_corp=False, operations=None, modules=None):
        request = dict(
            uid=uid,
            from_ts=from_ts,
            to_ts=to_ts,
        )
        if limit is not None:
            request['limit'] = limit
        if is_corp:
            request['corp'] = 'true'
        if operations is not None:
            request['operation'] = ','.join(operations)
        if modules is not None:
            request['module'] = ','.join(modules)
        return self._request_builder('/mail/2/user_history/', request)

    def build_last_letter_request(self, max_age):
        request = dict(
            max_age=max_age,
        )
        return self._request_builder('/sender/2/last_letter/', request)

    def push_history_by_fields(
        self, from_ts, to_ts, uid=None, device=None, app=None,
        subscription=None, limit=None,
    ):
        """
        Получение истории пушей по uid или device_id.
        """
        assert uid is not None or device is not None
        get_args = dict(
            from_ts=from_ts,
            to_ts=to_ts,
        )
        if uid is not None:
            get_args['uid'] = uid
        if device is not None:
            get_args['device'] = device
        if app is not None:
            get_args['app'] = app
        if subscription is not None:
            get_args['subscription'] = subscription
        if limit is not None:
            get_args['limit'] = limit
        request = self._request_builder('/push/2/by_fields/', get_args)
        return self._request_with_retries(
            'GET',
            request,
            forward_content(parse_push_history),
        )

    def push_history_by_push_id(
        self, push, subscription=None, uid=None, device=None, app=None,
    ):
        get_args = dict(
            push=push,
        )
        if uid is not None:
            get_args['uid'] = uid
        if device is not None:
            get_args['device'] = device
        if app is not None:
            get_args['app'] = app
        if subscription is not None:
            get_args['subscription'] = subscription
        request = self._request_builder('/push/2/by_push_id/', get_args)
        return self._request_with_retries(
            'GET',
            request,
            forward_content(parse_push_history),
        )


def get_historydb_api():
    return HistoryDBApi()  # pragma: no cover


__all__ = (
    'get_historydb_api',
    'HistoryDBApi',
    'ORDER_BY_TIMESTAMP_ASCENDING',
)
