# -*- coding: utf-8 -*-

from itertools import chain
import json

from passport.backend.core import authtypes
from passport.backend.core.builders.base.faker.fake_builder import BaseFakeBuilder
from passport.backend.core.builders.historydb_api.build_requests import HistoryDBApi
from passport.backend.utils.common import remove_none_values
from six.moves.urllib.parse import urlparse


_url_path_to_method_name = {
    '/2/auths/': 'auths',
    '/2/auths/aggregated/': 'auths_aggregated_old',
    '/3/auths/aggregated/': 'auths_aggregated',
    '/2/events/': 'events',
    '/2/events/restore/': 'events_restore',
    '/2/events/passwords/': 'events_passwords',
    '/2/auths/aggregated/runtime/': 'auths_aggregated_runtime',
    '/2/auths/failed/': 'auths_failed',
    '/2/lastauth/': 'lastauth',
    '/2/lastauth/bulk/': 'lastauth_bulk',
    '/mail/2/user_history/': 'mail_history',
    '/sender/2/last_letter/': 'last_letter',
    '/push/2/by_fields/': 'push_history_by_fields',
    '/push/2/by_push_id/': 'push_history_by_push_id',
}

method_names = {method for method in _url_path_to_method_name.values()}


TEST_YANDEXUID = '123'
TEST_UID = 123
TEST_UNIXTIME = 10000.5


class FakeHistoryDBApi(BaseFakeBuilder):
    service_name_template = 'historydb_api_%s_response'

    def __init__(self):
        super(FakeHistoryDBApi, self).__init__(HistoryDBApi)

    def set_response_value(self, method, response, status=200):
        if method not in method_names:
            raise ValueError(
                'Set response value for unknown method: {method}'.format(
                    method=method,
                ),
            )
        super(FakeHistoryDBApi, self).set_response_value(method, response, status)

    def set_response_side_effect(self, method, side_effect):
        if method not in method_names:
            raise ValueError(
                'Set response side effect for unknown method: {method}'.format(
                    method=method,
                ),
            )
        super(FakeHistoryDBApi, self).set_response_side_effect(
            method,
            side_effect,
        )

    @staticmethod
    def parse_method_from_request(http_method, url, data, headers=None):
        parsed_url = urlparse(url)
        return _url_path_to_method_name[parsed_url.path]


def event_item(client_name='passport',
               host_id=15,
               name='info.firstname',
               value='firstname',
               timestamp=3600,
               user_ip='87.250.235.4',
               yandexuid=TEST_YANDEXUID,
               ip_as_list=None,
               ip_geoid=None,
               **kwargs):
    item = dict(
        kwargs,
        client_name=client_name,
        host_id=host_id,
        name=name,
        value=value,
        timestamp=timestamp,
        user_ip=user_ip,
        yandexuid=yandexuid,
    )
    if ip_as_list is not None:
        item['ip.as_list'] = ip_as_list
    if ip_geoid is not None:
        item['ip.geoid'] = ip_geoid
    return remove_none_values(item)


def phone_event_item(phone_id=1, attribute='bound', **kwargs):
    return event_item(name='phone.%s.%s' % (phone_id, attribute), **kwargs)


def auth_successful_aggregated_browser_info(name='Firefox', version='33.0', yandexuid=None):
    return remove_none_values(
        dict(
            name=name,
            version=version,
            yandexuid=yandexuid,
        ),
    )


def auth_successful_aggregated_os_info(name='Windows 7', version='6.1'):
    return remove_none_values(
        dict(
            name=name,
            version=version,
        ),
    )


def auth_successful_aggregated_oauth_info(client_id='7a54f58d4ebe431caaaa53895522bf2d',
                                          scopes='cloud_api:disk.write,cloud_api:disk.app_folder',
                                          token_id='123',
                                          device_id='device_id',
                                          device_name=u'имя+девайса',
                                          AP=None):
    return remove_none_values(
        dict(
            client_id=client_id,
            scopes=scopes,
            token_id=token_id,
            device_id=device_id,
            device_name=device_name,
            AP=AP,
        ),
    )


def auth_successful_aggregated_ip_info(AS=13238, geoid=9999, ip='2a02:6b8:0:101:19c3:e71d:2e1d:5017'):
    return remove_none_values(
        dict(
            AS=AS,
            geoid=geoid,
            value=ip,
        ),
    )


def auth_successful_aggregated_runtime_ip_info(AS=13238, geoid=9999, ip='2a02:6b8:0:101:19c3:e71d:2e1d:5017'):
    return remove_none_values(
        dict(
            AS=AS,
            geoid=geoid,
            ip=ip,
        ),
    )


def auth_aggregated_browser_info(name='Firefox', version='33.0', yandexuid=None):
    return auth_successful_aggregated_browser_info(name=name, version=version, yandexuid=yandexuid)


def auth_aggregated_os_info(name='Windows 7', version='6.1'):
    return auth_successful_aggregated_os_info(name=name, version=version)


def auth_aggregated_oauth_info(client_id='7a54f58d4ebe431caaaa53895522bf2d',
                               scopes='cloud_api:disk.write,cloud_api:disk.app_folder',
                               token_id='123',
                               device_id='device_id',
                               device_name=u'имя+девайса',
                               AP=None):
    return remove_none_values(
        dict(
            clientId=client_id,
            scopes=scopes,
            tokenId=token_id,
            deviceId=device_id,
            deviceName=device_name,
            AP=AP,
        ),
    )


def auth_aggregated_ip_info(AS=13238, geoid=9999, ip='2a02:6b8:0:101:19c3:e71d:2e1d:5017'):
    return remove_none_values(
        dict(
            AS=AS,
            geoid=geoid,
            ip=ip,
        ),
    )


def auth_aggregated_item_old(ts, authtype, ip_info=None, browser_info=None, os_info=None, oauth_info=None):
    return dict(
        timestamps=[ts],
        count=1,
        auth=remove_none_values(dict(
            authtype=authtype,
            ip=ip_info,
            browser=browser_info,
            os=os_info,
            token=oauth_info,
        )),
    )


def auth_aggregated_item(ts, authtype, ip_info=None, browser_info=None, os_info=None, oauth_info=None):
    return dict(
        authentications=[{'timestamp': ts}],
        count=1,
        auth=remove_none_values(dict(
            authtype=authtype,
            ip=ip_info,
            browser=browser_info,
            os=os_info,
            token=oauth_info,
        )),
    )


def auth_successful_aggregated_runtime_auth_item(ip_info=None, browser_info=None, os_info=None, authtype=authtypes.AUTH_TYPE_WEB,
                                                 status='ses_create', count=1):
    """
    Возможные статусы - ses_create, ses_update, successful; типы - imap, oauthcheck, web...
    """
    auth_value = dict(
        authtype=authtype,
        status=status,
        ip=ip_info or {},
        browser=browser_info or {},
        os=os_info or {},
    )
    return dict(
        auth=auth_value,
        count=count,
    )


def auth_successful_aggregated_runtime_auths_item(timestamp=1418256000, auth_items=None):
    auths = auth_items or [auth_successful_aggregated_runtime_auth_item()]
    return dict(
        timestamp=timestamp,
        auths=auths,
    )


def auth_item(client_name='passport', host_id=15, timestamp=3600, user_ip='87.250.235.4',
              authtype=authtypes.AUTH_TYPE_WEB, status='successful', comment=None):
    return remove_none_values(dict(
        client_name=client_name,
        host_id=host_id,
        timestamp=timestamp,
        user_ip=user_ip,
        type=authtype,
        status=status,
        comment=comment,
    ))


def auth_failed_item(status='failed', **kwargs):
    return auth_item(status=status, **kwargs)


def mail_history_item(operation='mark', state='read;2160000000004499501', user_ip='87.250.235.4', **kwargs):
    item = {
        'affected': '1',
        'connectionId': '2c84bc9e7c2bd478f3fa82de5f1bd0be',
        'date': 1414666675947,
        'hidden': '0',
        'ip': user_ip,
        'mdb': 'mdb302',
        'module': 'mailbox_oper',
        'operation': operation,
        'regionId': '0',
        'state': state,
        'suid': '1120000000348711',
        'target': 'message',
        'yandexuidCookie': '7207633351394011299',
    }
    item.update(kwargs)
    return item


def auths_successful_aggregated_runtime_response(uid=TEST_UID, status='ok', items=None):
    items = items or []
    prev_timestamp = None
    for i, item in enumerate(items):
        item_timestamp = item['timestamp']
        if prev_timestamp is not None:
            if prev_timestamp <= item_timestamp:
                raise ValueError('Incorrect order of items at item %d' % i)
        prev_timestamp = item_timestamp
    response = dict(
        status=status,
        uid=uid,
        history=items,
    )
    return json.dumps(response).encode('utf-8')


def events_response(uid=TEST_UID, events=None, ascending=True):
    if events is None:
        events = [
            event_item(),
        ]
    if sorted(events, key=lambda item: item['timestamp'], reverse=not ascending) != events:
        raise ValueError('Incorrect order of items')
    return json.dumps({
        'events': events,
        'status': 'ok',
        'uid': uid,
    }).encode('utf-8')


def event_restore_item(timestamp=1234, restore_id='7E,13079,1408955588.53,3000453634,track_id',
                       action='restore_semi_auto_request',
                       data_json='{"field":"value"}'):
    return {
        'timestamp': timestamp,
        'restore_id': restore_id,
        'action': action,
        'data_json': data_json,
    }


def events_restore_response(uid=TEST_UID, restore_events=None, ascending=False):
    restore_events = restore_events if restore_events is not None else [event_restore_item()]
    if sorted(restore_events, key=lambda item: item['timestamp'], reverse=not ascending) != restore_events:
        raise ValueError('Incorrect order of items')
    return json.dumps({
        'uid': uid,
        'status': 'ok',
        'restore_events': restore_events,
    }).encode('utf-8')


def events_passwords_response(password_found=False, active_ranges=None):
    # важная особенность: ручка отдает список интервалов, отсортированных в следующем порядке:
    # сначала новые интервалы, потом старые
    active_ranges = active_ranges or []
    flat_ranges = list(chain(*[(end, start) for (start, end) in active_ranges]))
    if sorted(flat_ranges, reverse=True, key=lambda ts: (ts is None, ts)) != flat_ranges:
        # первым элементом при получении плоского списка может стать None - учитываем это ключом сортировки
        raise ValueError('Incorrect order of active ranges')
    return json.dumps({
        'status': 'ok',
        'password_found': password_found,
        'active_ranges': active_ranges or [],
    }).encode('utf-8')


def auths_response(uid=TEST_UID, auths=None):
    if auths is None:
        auths = [auth_item()]

    return json.dumps({
        'status': 'ok',
        'uid': uid,
        'auths': sorted(auths, key=lambda x: x['timestamp'], reverse=True),
    }).encode('utf-8')


def lastauth_response(uid=TEST_UID, _type=authtypes.AUTH_TYPE_WEB, timestamp=TEST_UNIXTIME):
    if _type is not None:
        lastauth = {'type': _type, 'timestamp': float(timestamp)}
    else:
        lastauth = {}
    return json.dumps({
        'status': 'ok',
        'uid': uid,
        'lastauth': lastauth,
    }).encode('utf-8')


def lastauth_bulk_response(uid_to_ts):
    if uid_to_ts is None:
        uid_to_ts = {TEST_UID: TEST_UNIXTIME}
    return json.dumps({
        'status': 'ok',
        'lastauth': {
            str(uid): {} if ts is None else {'timestamp': float(ts)}
            for uid, ts in uid_to_ts.items()
        },
    }).encode('utf-8')


def auths_aggregated_response_old(uid=TEST_UID, auths=None, next=None):
    if auths is None:
        auths = [
            auth_aggregated_item_old(
                authtype=authtypes.AUTH_TYPE_WEB,
                ip_info=auth_aggregated_ip_info(),
                browser_info=auth_aggregated_browser_info(),
                os_info=auth_aggregated_os_info(),
                ts=3600,
            ),
        ]

    return json.dumps({
        'status': 'ok',
        'uid': uid,
        'next': next,
        'auths': sorted(auths, key=lambda x: x['timestamps'][0], reverse=True),
    }).encode('utf-8')


def auths_aggregated_response(uid=TEST_UID, auths=None, next=None):
    if auths is None:
        auths = [
            auth_aggregated_item(
                authtype=authtypes.AUTH_TYPE_WEB,
                ip_info=auth_aggregated_ip_info(),
                browser_info=auth_aggregated_browser_info(),
                os_info=auth_aggregated_os_info(),
                ts=3600,
            ),
        ]

    return json.dumps({
        'status': 'ok',
        'uid': uid,
        'next': next,
        'auths': sorted(auths, key=lambda x: x['authentications'][0]['timestamp'], reverse=True),
    }).encode('utf-8')


def auths_failed_response(uid=TEST_UID, auths=None):
    if auths is None:
        auths = [auth_failed_item()]

    return json.dumps({
        'status': 'ok',
        'uid': uid,
        'auths': sorted(auths, key=lambda x: x['timestamp'], reverse=True),
    }).encode('utf-8')


def mail_history_response(uid=TEST_UID, items=None):
    if items is None:
        items = [mail_history_item()]

    return json.dumps({
        'status': 'ok',
        'uid': uid,
        'items': sorted(items, key=lambda x: x['date'], reverse=True),
    }).encode('utf-8')


def last_letter_response(uid=TEST_UID, list_id_to_ts=None):
    return json.dumps({
        'status': 'ok',
        'uid': uid,
        'items': {
            str(list_id): {'email_sent_ts': ts}
            for list_id, ts in list_id_to_ts.items()
        },
    }).encode('utf-8')


def push_history_response_item(
    app_id, push_id, unixtime, device_id, details, status, push_event,
    push_service, context, subscription_id, uid=TEST_UID,
):
    return dict(
        uid=uid,
        app_id=app_id,
        push_id=push_id,
        unixtime=unixtime,
        device_id=device_id,
        details=details,
        status=status,
        push_event=push_event,
        push_service=push_service,
        context=context,
        subscription_id=subscription_id,
    )


def push_history_response(items):
    return json.dumps({
        'status': 'ok',
        'items': items,
    })


def error_response(errors=['internal']):
    return json.dumps({'status': 'error', 'errors': errors}).encode('utf-8')


def events_info_interval_point(user_ip='1.1.1.1', timestamp=1, user_agent=None, yandexuid=TEST_YANDEXUID):
    values = {'user_ip': user_ip, 'timestamp': timestamp, 'user_agent': user_agent, 'yandexuid': yandexuid}
    return {field: value for field, value in values.items() if field == 'timestamp' or value is not None}


def app_key_info(device_ifv='66530E20-0BD7-4C46-9F57-88058D91A123',
                 device_hardware_id='65b32e0724f86f10377baf613489d123', is_ios=False):
    if is_ios:
        data = {
            'device_ifv': device_ifv,
            'device_app_uuid': '',
            'device_application_version': '103',
            'device_hardware_model': 'iPhone',
            'device_application': 'ru.yandex.mobile.kluch',
            'device_os_id': 'iOS',
            'device_manufacturer': 'Apple',
            'device_name': 'iPhone 6+',
        }
    else:
        data = {
            'device_app_uuid': '8444b0cb9f2cb52cdbc3c45367ac7123',
            'device_application_version': '1.04',
            'device_hardware_model': 'GT-P7500',
            'device_hardware_id': device_hardware_id,
            'device_application': 'ru.yandex.key',
            'device_os_id': 'android',
            'device_manufacturer': 'samsung',
        }
    return json.dumps(data).encode('utf-8')
