# -*- coding: utf-8 -*-
# Описание ручек API на Вики https://wiki.yandex-team.ru/passport/historydb/api
import json
import logging

from passport.backend.core.builders.historydb_api.exceptions import (
    HistoryDBApiInvalidResponseError,
    HistoryDBApiPermanentError,
    HistoryDBApiTemporaryError,
)
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.utils.string import smart_text


log = logging.getLogger('passport.historydb_api.parsers')


def errors_in_historydb_api_response(response):
    if 'status' not in response:
        raise HistoryDBApiInvalidResponseError('Response without status: %s' % response)
    if response['status'] == 'error':
        raise HistoryDBApiTemporaryError(response['errors'])
    return False


def errors_in_historydb_api_response_permanent(response):
    if 'status' not in response:
        raise HistoryDBApiInvalidResponseError(
            'Response without status: %s' % response,
        )
    if response['status'] == 'error':
        raise HistoryDBApiPermanentError(response['errors'])
    return False


def parse_historydb_api_json(response):
    try:
        return json.loads(response)
    except ValueError:
        log.warning('HistoryDBApi returned invalid JSON: %s', trim_message(smart_text(response)))
        raise HistoryDBApiInvalidResponseError('Invalid JSON in HistoryDBApi response.')


def parse_historydb_api_events(response):
    response = parse_historydb_api_json(response)

    errors_in_historydb_api_response(response)
    return response['events']


def parse_historydb_api_events_restore(response):
    response = parse_historydb_api_json(response)

    errors_in_historydb_api_response(response)
    return response['restore_events']


def parse_historydb_api_events_passwords(response):
    response = parse_historydb_api_json(response)

    errors_in_historydb_api_response(response)
    return response['password_found'], response['active_ranges']


def parse_auths(response):
    response = parse_historydb_api_json(response)

    errors_in_historydb_api_response(response)
    return response['auths']


def parse_auths_aggregated(response):
    response = parse_historydb_api_json(response)

    errors_in_historydb_api_response(response)
    return response['auths'], response['next']


def parse_auths_aggregated_runtime(response):
    response = parse_historydb_api_json(response)

    errors_in_historydb_api_response(response)
    return response


def parse_auths_failed(response):
    response = parse_historydb_api_json(response)

    errors_in_historydb_api_response(response)
    return response['auths']


def parse_lastauth(response):
    response = parse_historydb_api_json(response)
    errors_in_historydb_api_response(response)
    return response['lastauth']


def parse_lastauth_bulk(response):
    response = parse_historydb_api_json(response)
    errors_in_historydb_api_response_permanent(response)
    return {
        int(uid): item.get('timestamp')
        for uid, item in response['lastauth'].items()
    }


def parse_mail_history(response):
    response = parse_historydb_api_json(response)
    errors_in_historydb_api_response_permanent(response)
    return response['items']


def parse_last_letter(response):
    response = parse_historydb_api_json(response)
    errors_in_historydb_api_response_permanent(response)
    return {
        int(list_id): item['email_sent_ts']
        for list_id, item in response['items'].items()
    }


def parse_push_history(response):
    response = parse_historydb_api_json(response)
    errors_in_historydb_api_response_permanent(response)
    return response['items']


__all__ = (
    'parse_historydb_api_events',
    'parse_historydb_api_events_passwords',
    'parse_historydb_api_events_restore',
    'parse_auths',
    'parse_auths_aggregated',
    'parse_auths_aggregated_runtime',
    'parse_auths_failed',
    'parse_lastauth',
    'parse_lastauth_bulk',
    'parse_last_letter',
    'parse_mail_history',
)
