# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import (
    BaseBuilder,
    parser_trivial,
)
from passport.backend.core.builders.kolmogor.exceptions import (
    BaseKolmogorError,
    KolmogorBadRequestError,
    KolmogorPermanentError,
    KolmogorTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.kolmogor')


def kolmogor_error_detector(response, raw_response):
    if raw_response.status_code == 200:
        return
    if raw_response.status_code == 400:
        message = raw_response.content.replace(b'Error:', b'').strip().decode('utf-8')
        raise KolmogorBadRequestError(message)

    log.warning(
        u'Request failed with response=%s code=%s',
        trim_message(raw_response.content.decode('utf-8')),
        raw_response.status_code,
    )
    raise KolmogorPermanentError('status=%s' % raw_response.status_code)


class Kolmogor(BaseBuilder):
    base_error_class = BaseKolmogorError
    temporary_error_class = KolmogorTemporaryError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, graphite_logger=None,
                 use_tvm=False, **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='kolmogor')
        super(Kolmogor, self).__init__(
            url=url or settings.KOLMOGOR_URL,
            timeout=timeout or settings.KOLMOGOR_TIMEOUT,
            retries=retries or settings.KOLMOGOR_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias='kolmogor' if use_tvm else None,
            **kwargs
        )

    def get(self, space, keys):
        keys = sorted(set(keys))
        resp = self._request_with_retries_simple(
            url_suffix='get',
            method='GET',
            params=dict(
                space=space,
                keys=','.join(keys),
            ),
            parser=parser_trivial,
            error_detector=kolmogor_error_detector,
        )
        values = [int(value) for value in resp.strip().split(',')]
        if len(keys) != len(values):
            raise KolmogorPermanentError('Received wrong values (%s) for keys %s' % (values, keys))
        return dict(zip(keys, values))

    def inc(self, space, keys):
        self._request_with_retries_simple(
            url_suffix='inc',
            method='POST',
            data=dict(
                space=space,
                keys=','.join(keys),
            ),
            parser=parser_trivial,
            error_detector=kolmogor_error_detector,
        )


def get_kolmogor():
    return Kolmogor()  # pragma: no cover
