# -*- coding: utf-8 -*-
# Документация по API поиска в адресной книге: https://pages.github.yandex-team.ru/ilya-sidorov/contacts/
import logging

from passport.backend.core.builders.mail_apis.base import BaseMailApiBuilder
from passport.backend.core.builders.mail_apis.exceptions import (
    BaseCollieError,
    CollieInvalidResponseError,
    CollieTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.mail_apis.collie')


DEFAULT_LIMIT = 500


class Collie(BaseMailApiBuilder):
    """
    Билдер для общения с сервисом Колли.
    """
    base_error_class = BaseCollieError
    temporary_error_class = CollieTemporaryError
    parser_error_class = CollieInvalidResponseError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None,
                 graphite_logger=None):
        graphite_logger = graphite_logger or GraphiteLogger(service='collie')
        super(Collie, self).__init__(
            url=url or settings.COLLIE_API_URL,
            timeout=timeout or settings.COLLIE_API_TIMEOUT,
            retries=retries or settings.COLLIE_API_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias='collie',
        )

    def _parse_contacts(self, raw_response):
        data = self.parse_json(raw_response)
        return data.get('contacts', [])

    def search_contacts(self, uid, offset=0, limit=DEFAULT_LIMIT, request_id=None):
        """
        Поиск контактов. В текущем варианте отдаются email-адреса без группировки по контактам.
        """
        params = {
            'offset': offset,
            'limit': limit,
        }
        headers = {}
        if request_id:
            headers['X-Request-Id'] = request_id

        return self._request_with_retries_simple(
            url_suffix='/v1/users/%s/contacts/[]' % uid,
            http_error_handler=self.handle_http_error,
            error_detector=None,
            parser=self._parse_contacts,
            params=params,
            headers=headers,
        )


def get_collie(timeout=None):
    return Collie(timeout=timeout)


__all__ = [
    'get_collie',
    'Collie',
]
