# -*- coding: utf-8 -*-
import json
import logging

from passport.backend.core.builders.mail_apis.base import BaseMailApiBuilder
from passport.backend.core.builders.mail_apis.exceptions import (
    BaseHuskyError,
    HuskyInvalidResponseError,
    HuskyTaskAlreadyExistsError,
    HuskyTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.utils.string import smart_text


log = logging.getLogger('passport.mail_apis.husky')


def husky_api_delete_user_error_detector(data, raw_response):
    if 'error' in data:
        code = int(data['error'].get('code', 1))
        if code == 1:
            raise HuskyTemporaryError(data['error'].get('message'))
        elif code in [2, 4]:
            raise HuskyTaskAlreadyExistsError()
        else:
            message = u'HuskyApi error response (status=%s code=%s message=%s)' % (
                data.get('status'),
                data['error']['code'],
                smart_text(trim_message(data['error'].get('message'))),
            )
            raise HuskyInvalidResponseError(message)
    if not data.get('tasks') or raw_response.status_code != 200:
        message = u'Husky invalid response: no tasks (http_status=%s status=%s data=%s)' % (
            raw_response.status_code,
            data.get('status'),
            trim_message(json.dumps(data)),
        )
        raise HuskyInvalidResponseError(message)


class HuskyApi(BaseMailApiBuilder):
    """
    Билдер для общения с сервисом Husky.
    Документация: https://wiki.yandex-team.ru/pochta/backend/husky-api/
    """
    base_error_class = BaseHuskyError
    temporary_error_class = HuskyTemporaryError
    parser_error_class = HuskyInvalidResponseError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, graphite_logger=None):
        timeout = timeout or settings.HUSKY_API_TIMEOUT
        retries = retries or settings.HUSKY_API_RETRIES
        graphite_logger = graphite_logger or GraphiteLogger(service='husky')
        super(HuskyApi, self).__init__(
            url=url or settings.HUSKY_API_URL,
            timeout=timeout,
            retries=retries,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias='husky',
        )

    def delete_user(self, uid):
        """
        Удаление почтовых метаданных пользователя.
        """
        return self._request_with_retries_simple(
            method='POST',
            url_suffix='delete_user/%s' % uid,
            error_detector=husky_api_delete_user_error_detector,
            parser=self.parse_json,
        )


def get_husky():
    return HuskyApi()    # pragma: no cover


def drop_mailbox(uid):
    if settings.HUSKY_ENABLED:
        # ставим задачу в Husky на удаление
        try:
            get_husky().delete_user(uid)
        except HuskyTaskAlreadyExistsError:
            pass
