# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.mail_apis.base import BaseMailApiBuilder
from passport.backend.core.builders.mail_apis.exceptions import (
    BaseWMIError,
    WMIInvalidResponseError,
    WMITemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.mail_apis.wmi')


class WMI(BaseMailApiBuilder):
    """
    Билдер для общения с сервисом WMI (Hound в терминах Почты).
    https://wiki.yandex-team.ru/users/jkennedy/ywmiapi
    """
    base_error_class = BaseWMIError
    temporary_error_class = WMITemporaryError
    parser_error_class = WMIInvalidResponseError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None,
                 graphite_logger=None):
        timeout = timeout or settings.WMI_API_TIMEOUT
        retries = retries or settings.WMI_API_RETRIES
        graphite_logger = graphite_logger or GraphiteLogger(service='wmi')
        super(WMI, self).__init__(
            url=url or settings.WMI_API_URL,
            timeout=timeout,
            retries=retries,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias='wmi_api',
        )

    def folders(self, uid, suid, mdb):
        """
        Получение папок пользователя.
        """
        params = {
            'uid': uid,
            'suid': suid,
            'mdb': mdb,
        }

        return self._request_with_retries_simple(
            url_suffix='folders',
            http_error_handler=self.handle_http_error,
            error_detector=error_detector,
            parser=self.parse_json,
            params=params,
        )


def error_detector(data, raw_response):
    if 'error' in data:
        raise WMIInvalidResponseError(data['error'])


def get_wmi(timeout=None):
    return WMI(timeout=timeout)


__all__ = [
    'get_wmi',
    'WMI',
]
