# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.market.exceptions import (
    BaseMarketContentApiError,
    MarketContentApiInvalidResponseError,
    MarketContentApiPermanentError,
    MarketContentApiTemporaryError,
)
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.core.utils.domains import get_keyspace_by_host


log = logging.getLogger('passport.market.content_api')


MARKET_CONTENT_API_VERSION = 'v2.1.0'
MARKET_REQUEST_ID_HEADER = 'X-Market-Req-ID'
USER_ORDERS_FAKE_UUID = '01234567012345670123456701234567'


def market_http_error_handler(raw_response):
    if raw_response.status_code in (502, 503):
        req_id = raw_response.headers.get(MARKET_REQUEST_ID_HEADER, 'missing in response')
        log.warning(
            u'Request failed with response=%s code=%s req_id=%s',
            trim_message(raw_response.content.decode('utf-8')),
            raw_response.status_code,
            req_id,
        )
        raise MarketContentApiTemporaryError()


class MarketContentApi(BaseBuilder, JsonParserMixin):
    """
    Билдер для общения с Контентным API Маркета
    Подробнее: https://tech.yandex.ru/market/monetization/doc/dg-v2/concepts/about-docpage/
    """

    base_error_class = BaseMarketContentApiError
    temporary_error_class = MarketContentApiTemporaryError
    parser_error_class = MarketContentApiInvalidResponseError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None,
                 graphite_logger=None, api_key=None):
        timeout = timeout or settings.MARKET_CONTENT_API_TIMEOUT
        retries = retries or settings.MARKET_CONTENT_API_RETRIES
        self.api_key = api_key or settings.MARKET_CONTENT_API_KEY
        graphite_logger = graphite_logger or GraphiteLogger(service='market_content_api')
        super(MarketContentApi, self).__init__(
            url=url or settings.MARKET_CONTENT_API_URL,
            timeout=timeout,
            retries=retries,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
        )

    def check_response_for_errors(self, data, raw_response):
        request_id = raw_response.headers.get(MARKET_REQUEST_ID_HEADER, 'missing in response')
        message_prefix = 'Request id: {}; '.format(request_id)
        if (
            raw_response.status_code == 422 and
            data.get('message', '').find('\'geo_id\' is missing') >= 0
        ):
            # https://st.yandex-team.ru/PASSP-21936
            raise MarketContentApiPermanentError(message_prefix + str(data.get('message')))
        if (
            raw_response.status_code == 500 and
            'internal error' in data.get('message', '').lower()
        ):
            # https://st.yandex-team.ru/PASSP-26158
            raise MarketContentApiTemporaryError(message_prefix + str(data.get('message')))

        if 'errors' in data:
            error_message = '%s%s (status %s)' % (
                message_prefix,
                str(data['errors']),
                raw_response.status_code,
            )
            if raw_response.status_code >= 500:
                raise MarketContentApiPermanentError(error_message)
            raise MarketContentApiInvalidResponseError(error_message)

    def _make_request(self, url_suffix='', method='GET', oauth_token=None,
                      sessionid=None, host=None, params=None, count=None, page=None):

        params = params or {}

        if count is not None:
            if count < 1 or count > 50:
                raise ValueError('Incorrect count value')
            params['count'] = count
        if page is not None:
            if page < 1:
                raise ValueError('Incorrect page value')
            params['page'] = page

        headers = {
            'Authorization': self.api_key,
        }

        if oauth_token:
            headers['X-User-Authorization'] = 'OAuth {}'.format(oauth_token)
        elif sessionid:
            if not host:
                raise ValueError('Host is required when sessionid is specified')
            # TODO: перед включением sessguard для этого сервиса переделать передачу кук и хоста (PASSP-30037)
            host = get_keyspace_by_host(host)
            headers['X-User-Authorization'] = 'SessionId Id={}; Host={}'.format(sessionid, host)
        else:
            raise ValueError('User credential missing')

        return self._request_with_retries_simple(
            method=method,
            url_suffix=url_suffix,
            error_detector=self.check_response_for_errors,
            parser=self.parse_json,
            http_error_handler=market_http_error_handler,
            params=params,
            headers=headers,
        )

    def user_orders(self, oauth_token=None, sessionid=None, host=None, fields=None, from_date=None,
                    to_date=None, count=None, page=None):
        params = {
            'format': 'json',
            # Ручка изначально делалась для мобильных и без UUID не работает
            'uuid': USER_ORDERS_FAKE_UUID,
        }
        if fields is not None:
            params['fields'] = ','.join(fields)
        if from_date is not None:
            params['from_date'] = from_date.strftime('%Y-%m-%d')
        if to_date is not None:
            params['to_date'] = to_date.strftime('%Y-%m-%d')

        return self._make_request(
            url_suffix='user/orders',
            oauth_token=oauth_token,
            sessionid=sessionid,
            host=host,
            params=params,
            count=count,
            page=page,
        )

    def user_wishlist(self, user_ip, fields=None, sort_by='create_date', how='desc',
                      oauth_token=None, sessionid=None, host=None, count=None, page=None):
        params = {
            'remote_ip': user_ip,
            'sort': sort_by,
            'how': how,
        }
        if fields is not None:
            params['fields'] = ','.join(fields)
        return self._make_request(
            url_suffix='%s/user/wishlist/items' % MARKET_CONTENT_API_VERSION,
            oauth_token=oauth_token,
            sessionid=sessionid,
            host=host,
            params=params,
            count=count,
            page=page,
        )


def get_market_content_api(timeout=None):
    return MarketContentApi(timeout=timeout)  # pragma: no cover


__all__ = [
    'get_market_content_api',
    'MarketContentApi',
    'MARKET_CONTENT_API_VERSION',
]
