# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.meltingpot_api.exceptions import (
    BaseMeltingpotApiError,
    MeltingpotApiInvalidResponseError,
    MeltingpotApiTemporaryError,
)
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.meltingpot_api')


class MeltingpotApi(BaseBuilder, JsonParserMixin):
    """
    Билдер для общения с сервисом Большой Котел
    """

    base_error_class = BaseMeltingpotApiError
    temporary_error_class = MeltingpotApiTemporaryError
    parser_error_class = MeltingpotApiInvalidResponseError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, consumer=None,
                 graphite_logger=None, tvm_dst_alias=None, tvm_credentials_manager=None):
        timeout = timeout or settings.MELTINGPOT_API_TIMEOUT
        retries = retries or settings.MELTINGPOT_API_RETRIES
        self.consumer = consumer or settings.MELTINGPOT_API_CONSUMER
        graphite_logger = graphite_logger or GraphiteLogger(service='meltingpot')
        super(MeltingpotApi, self).__init__(
            url=url or settings.MELTINGPOT_API_URL,
            timeout=timeout,
            retries=retries,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias=tvm_dst_alias,
            tvm_credentials_manager=tvm_credentials_manager,
        )

    def check_response_for_errors(self, data, raw_response):
        if data.get('status') != 'ok':
            raise self.base_error_class()

    def add_user(self, uid, reason, priority, group_id=None):
        params = {
            'consumer': self.consumer,
        }

        data = {
            'uid': uid,
            'reason': reason,
            'priority': priority,
        }

        if group_id:
            data['group_id'] = group_id

        return self._request_with_retries_simple(
            method='POST',
            url_suffix='1/users/',
            error_detector=self.check_response_for_errors,
            parser=self.parse_json,
            params=params,
            data=data,
        )

    def add_users_many(self, uids, reason, priority, group_id=None):
        params = {
            'consumer': self.consumer,
        }

        data = {
            'uids': ','.join(map(str, uids)),
            'reason': reason,
            'priority': priority,
        }

        if group_id:
            data['group_id'] = group_id

        return self._request_with_retries_simple(
            method='POST',
            url_suffix='1/users/many/',
            error_detector=self.check_response_for_errors,
            parser=self.parse_json,
            params=params,
            data=data,
        )

    def proxy(self, method, path, params=None, data=None, files=None):
        params = params or {}
        params['consumer'] = self.consumer
        return self._request_with_retries_simple(
            url_suffix=path,
            error_detector=None,
            parser=self.parse_json,
            method=method,
            params=params,
            data=data,
            files=files,
        )


def get_meltingpot_api(timeout=None):
    return MeltingpotApi(timeout=timeout)  # pragma: no cover


__all__ = [
    'get_meltingpot_api',
    'MeltingpotApi',
]
