# coding: utf-8
import logging
from time import time

from mssngr.router.lib.protos import message_pb2
from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.messenger_api.exceptions import (
    BaseMessengerApiError,
    MessengerApiPermanentError,
    MessengerApiTemporaryError,
)
from passport.backend.core.builders.messenger_api.serializer import (
    parse,
    ParsingError,
    serialize,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.messenger_fanout_api')


def messenger_api_http_error_handler(response):
    if response.status_code in (502, 503):
        raise MessengerApiTemporaryError(u'Request failed with code=%s' % response.status_code)

    if response.status_code != 200:
        raise MessengerApiPermanentError(u'Request failed with code=%s' % response.status_code)


class MssngrFanoutAPI(BaseBuilder):
    """
    API мессенджера, в том числе для проверки последнего появления пользователя online
    https://wiki.yandex-team.ru/messenger/fanoutapi/
    """
    base_error_class = BaseMessengerApiError
    temporary_error_class = MessengerApiTemporaryError
    parser_error_class = MessengerApiPermanentError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, graphite_logger=None,
                 **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='messenger_api')
        super(MssngrFanoutAPI, self).__init__(
            url=url or settings.MESSENGER_FANOUT_API_URL,
            timeout=timeout or settings.MESSENGER_FANOUT_API_TIMEOUT,
            retries=retries or settings.MESSENGER_FANOUT_API_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias='messenger_api',
            **kwargs
        )

    def check_response_for_errors(self, response, raw_response):
        if response.ErrorInfos:
            error_message = u'Message contain some errors: {}'.format(
                '; '.join([
                    u'WorkerId: {}, TimeMcs: {}, Error message: `{}`'.format(
                        error.WorkerId,
                        error.TimeMcs,
                        error.Message
                    ) for error in response.ErrorInfos])
            )
            raise MessengerApiPermanentError(error_message)
        if response.Status != 0:
            raise MessengerApiPermanentError(u'Unexpected status code {} without error in response'.format(response.Status))

    def response_parser(self, message):
        def _response_parser(response):
            try:
                parse(response.content, message)
            except ParsingError:
                raise self.parser_error_class()
            return message
        return _response_parser

    def _make_request(self, data, url_suffix, parser=None):
        response = self._request_with_retries_simple(
            error_detector=self.check_response_for_errors,
            parser=parser,
            method='POST',
            url_suffix=url_suffix,
            data=serialize(data),
            headers={
                'Content-Type': 'application/octet-stream'
            },
            http_error_handler=messenger_api_http_error_handler,
        )
        return response

    def check_user_lastseen(self, uid):
        now = time()
        request = message_pb2.TLastSeenRequest()
        request.Uids.extend([uid])

        response = self._make_request(data=request, url_suffix='last_seen', parser=self.response_parser(message_pb2.TLastSeenResponse()))
        if not response.LastSeenInfos:
            return -1  # никогда не был в мессенджере
        lastseeninfo = response.LastSeenInfos[0].Timestamp  # шлем 1 uid и это его utc время онлайн
        was_online_ago = now - lastseeninfo / 10 ** 6
        if was_online_ago < 0:  # был в будущем, возможно из-за рассинхронизации времени
            return 0
        return int(was_online_ago)  # сколько секунд назад юзер был онлайн


def get_mssngr_fanout_api():
    return MssngrFanoutAPI()  # pragma: no cover
