# coding: utf-8
import struct

import cityhash
from google.protobuf.message import DecodeError


class ParsingError(Exception):
    pass


def parse(data, message):
    try:
        version, checksum = struct.unpack('<LQ', data[0:12])
    except TypeError:
        raise ParsingError('Incorrect message header')
    # Запрос ушел с версией 2, в ответе ожидаем ее же
    # https://wiki.yandex-team.ru/messenger/fanoutapi/#osnovnyemomenty
    if version != 2:
        raise ParsingError('Incorrect version: %s' % version)
    tail = data[12:]
    expected_checksum = cityhash.hash64(tail)
    if checksum != expected_checksum:
        raise ParsingError('Incorrect checksum, got %s expected %s' % (checksum, expected_checksum))
    try:
        message.ParseFromString(tail)
    except DecodeError:
        raise ParsingError('Incorrect decompressed protobuf')
    return message


def serialize(message):
    data = message.SerializeToString()
    version = 2
    checksum = cityhash.hash64(data)
    header = struct.pack('<LQ', version, checksum)
    return header + data
