# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.builders.money_api.exceptions import (
    BaseMoneyApiError,
    MoneyApiInvalidResponseError,
    MoneyApiPermanentError,
    MoneyApiTemporaryError,
    MoneyApiUnknownSchemeError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.money_payment_auth_api')


PAYMENT_AUTH_SUCCEEDED_STATUS = 'Succeeded'


def money_error_detector(response, raw_response):
    if raw_response.status_code == 200:
        return

    if response is None:
        error_message = 'Empty response (status: %s)' % raw_response.status_code
        raise MoneyApiPermanentError(error_message)
    else:
        error_info = response.get('error', {})
        error_type = error_info.get('type', 'Unknown error')

        if raw_response.status_code == 404 and error_type == 'Scheme not found':
            raise MoneyApiUnknownSchemeError()

        error_message = '%s (status: %s)' % (error_type, raw_response.status_code)
        raise MoneyApiPermanentError(error_message)


class MoneyPaymentAuthApi(BaseBuilder, JsonParserMixin):
    """
    Билдер для общения с API Платёжной авторизации Денег
    Подробнее: https://st.yandex-team.ru/PASSP-20881#1540819890000
    """

    base_error_class = BaseMoneyApiError
    temporary_error_class = MoneyApiTemporaryError
    parser_error_class = MoneyApiInvalidResponseError
    accept_empty_response = True  # некоторые ошибки приходят с пустым телом

    def __init__(self, url=None, useragent=None, timeout=None, retries=None,
                 graphite_logger=None, **kwargs):
        timeout = timeout or settings.MONEY_PAYMENT_AUTH_API_TIMEOUT
        retries = retries or settings.MONEY_PAYMENT_AUTH_API_RETRIES
        graphite_logger = graphite_logger or GraphiteLogger(service='money_api')
        super(MoneyPaymentAuthApi, self).__init__(
            url=url or settings.MONEY_PAYMENT_AUTH_API_URL,
            timeout=timeout,
            retries=retries,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            **kwargs
        )

    def create_auth_context(self, uid, client_id, scopes, retpath, request_id, scheme=None):
        params = {}
        if scheme:
            params.update(scheme=scheme)
        data = {
            'uid': str(uid),
            'clientId': client_id,
            'scope': ' '.join(scopes),
            'returnUri': retpath,
            'passportRequestId': request_id,
        }

        return self._request_with_retries_simple(
            method='POST',
            url_suffix='v1/context',
            params=params,
            json_data=data,
            error_detector=money_error_detector,
            parser=self.parse_json,
        )

    def check_auth_context(self, auth_context_id, scheme=None):
        params = {
            'authContextId': auth_context_id,
        }
        if scheme:
            params.update(scheme=scheme)

        return self._request_with_retries_simple(
            method='GET',
            url_suffix='v1/context',
            params=params,
            error_detector=money_error_detector,
            parser=self.parse_json,
        )


def get_money_payment_auth_api():
    return MoneyPaymentAuthApi()  # pragma: no cover
