# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.builders.music_api.exceptions import (
    BaseMusicApiError,
    MusicApiInvalidResponseError,
    MusicApiPermanentError,
    MusicApiTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.utils.common import (
    merge_dicts,
    noneless_dict,
)


log = logging.getLogger('passport.music_api')


def music_api_error_detector(response, raw_response):
    if raw_response.status_code != 200:
        error_message = response.get('error', {}).get('message')
        raise MusicApiPermanentError(error_message)


def music_api_response_processor(response):
    return response['result']


class MusicApi(BaseBuilder, JsonParserMixin):
    """
    Билдер для общения с API Музыки
    Подробнее: https://doc.yandex-team.ru/music/api-guide/concepts/about.html
    """

    base_error_class = BaseMusicApiError
    temporary_error_class = MusicApiTemporaryError
    parser_error_class = MusicApiInvalidResponseError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None,
                 graphite_logger=None, **kwargs):
        timeout = timeout or settings.MUSIC_API_TIMEOUT
        retries = retries or settings.MUSIC_API_RETRIES
        graphite_logger = graphite_logger or GraphiteLogger(service='music_api')
        super(MusicApi, self).__init__(
            url=url or settings.MUSIC_API_URL,
            timeout=timeout,
            retries=retries,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias='music_api',
            **kwargs
        )

    @staticmethod
    def headers(user_ticket):
        return {
            'Accept': 'application/json',
            'X-Ya-User-Ticket': user_ticket,
        }

    def music_api_request(self, method, url_suffix, uid, user_ticket, **extra_params):
        """Запрос в API музыки по uid, с TVM-тикетом, парсит json"""
        params = {
            '__uid': uid,
        }
        data = {}
        params = noneless_dict(merge_dicts(params, extra_params))
        if method == 'POST':
            data = params
            params = {}
        return self._request_with_retries_simple(
            method=method,
            url_suffix=url_suffix,
            params=params,
            data=data,
            error_detector=music_api_error_detector,
            parser=self.parse_json,
            response_processor=music_api_response_processor,
            headers=self.headers(user_ticket),
        )

    def account_status(self, uid, user_ticket, country_id, user_ip):
        return self.music_api_request(
            'GET',
            'account/status',
            uid,
            user_ticket,
            region=country_id,
            ip=user_ip,
        )


def get_music_api():
    return MusicApi()  # pragma: no cover
