# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.exceptions import BaseCoreError
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.utils.common import remove_none_values
from six.moves.urllib.parse import urlparse


log = logging.getLogger('passport.oauth')

OAUTH_CODE_STRENGTH_LONG = 'long'

OAUTH_BUNDLE_API_INVALID_CLIENT_ID_ERRORS = {
    'client.not_found',
    'client_id.invalid',
    'client_id.too_long',
    'client_id.too_short',
}

OAUTH_BUNDLE_API_INVALID_CLIENT_SECRET_ERRORS = {
    'client_secret.not_matched',
    'client_secret.invalid',
    'client_secret.too_long',
    'client_secret.too_short',
}


OAUTH_INVALID_GRANT_ERROR = 'invalid_grant'
OAUTH_PAYMENT_AUTH_PENDING_ERROR = 'payment_auth_pending'

OAUTH_FORBIDDEN_ACCOUNT_TYPE_ERROR_DESCRIPTION = 'forbidden account type'


class BaseOAuthError(BaseCoreError):
    pass


class OAuthTemporaryError(BaseOAuthError):
    """Временная ошибка OAuth - стоит поретраиться"""


class OAuthPermanentError(BaseOAuthError):
    """Пятисотка или иная непредвиденная ошибка"""


class OAuthInvalidClientIdError(OAuthPermanentError):
    pass


class OAuthInvalidClientSecretError(OAuthPermanentError):
    pass


class OAuthAccountNotFoundError(OAuthPermanentError):
    pass


class OAuthDeviceCodeNotFound(OAuthPermanentError):
    pass


class OAuthDeviceCodeNotAccepted(OAuthPermanentError):
    pass


def oauth_error_handler(response):
    if response.status_code >= 500:
        log.warning(
            u'Request failed with with response=%s code=%s',
            trim_message(response.content.decode('utf-8')),
            response.status_code,
        )
        if response.status_code == 503:
            raise OAuthTemporaryError('Request failed')
        else:
            raise OAuthPermanentError('Server error')


def bundle_api_error_detector(response, raw_response):
    if response.get('errors'):
        error = response['errors'][0]
        if error == 'backend.failed':
            raise OAuthTemporaryError()
        elif error in OAUTH_BUNDLE_API_INVALID_CLIENT_ID_ERRORS:
            raise OAuthInvalidClientIdError()
        elif error in OAUTH_BUNDLE_API_INVALID_CLIENT_SECRET_ERRORS:
            raise OAuthInvalidClientSecretError()
        elif error == 'user.not_found':
            raise OAuthAccountNotFoundError()
        elif error == 'code.not_found':
            raise OAuthDeviceCodeNotFound()
        elif error == 'code.not_accepted':
            raise OAuthDeviceCodeNotAccepted()
        else:
            raise OAuthPermanentError(error)


class OAuth(BaseBuilder, JsonParserMixin):

    base_error_class = BaseOAuthError
    temporary_error_class = OAuthTemporaryError
    parser_error_class = OAuthPermanentError

    _USER_INFO_ARGS = {
        'user_ip',
        'am_version',
        'am_version_name',
        'app_id',
        'app_platform',
        'manufacturer',
        'model',
        'app_version',
        'app_version_name',
        'uuid',
        'deviceid',
        'ifv',
        'device_name',
        'os_version',
        'device_id',
    }

    def __init__(self, oauth=None, useragent=None, timeout=None, retries=None, graphite_logger=None, consumer=None,
                 ca_cert=None, tvm_dst_alias='oauth'):
        self.consumer = consumer or settings.OAUTH_CONSUMER
        timeout = timeout or settings.OAUTH_TIMEOUT
        graphite_logger = graphite_logger or GraphiteLogger(service='oauth')
        super(OAuth, self).__init__(
            url=oauth or settings.OAUTH_URL,
            timeout=timeout,
            retries=retries or settings.OAUTH_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            ca_cert=ca_cert or settings.SSL_CA_CERT,
            tvm_dst_alias=tvm_dst_alias,
        )

    def params_from_user_info_data(self, data):
        # Часть параметров (адрес пользователя, сведения об устройстве)
        # передаём в query, чтобы можно было связать логи этих запросов вместе.
        params = {}
        for ui_arg in self._USER_INFO_ARGS:
            if data.get(ui_arg):
                params[ui_arg] = data.pop(ui_arg)
        return params

    def _token(
        self, grant_type, client_id, client_secret, **kwargs
    ):
        params = self.params_from_user_info_data(kwargs)
        data = remove_none_values(dict(
            grant_type=grant_type,
            client_id=client_id,
            client_secret=client_secret,
            **kwargs
        ))
        return self._request_with_retries_simple(
            None,
            self.parse_json,
            url_suffix='token',
            method='POST',
            params=params,
            data=data,
            http_error_handler=oauth_error_handler,
        )

    def token_by_sessionid(self, client_id, client_secret, sessionid, host, user_ip, **kwargs):
        return self._token(
            'sessionid',
            client_id,
            client_secret,
            sessionid=sessionid,
            host=host,
            user_ip=user_ip,
            **kwargs
        )

    def token_by_uid(
        self, client_id, client_secret, uid, user_ip, password_passed=False,
        login_id=None, set_is_xtoken_trusted=None, **kwargs
    ):
        return self._token(
            'passport_assertion',
            client_id,
            client_secret,
            assertion=uid,
            user_ip=user_ip,
            password_passed=password_passed,
            login_id=login_id,
            set_is_xtoken_trusted=set_is_xtoken_trusted,
            **kwargs
        )

    def token_by_x_token(self, client_id, client_secret, x_token, user_ip, **kwargs):
        return self._token(
            'x-token',
            client_id,
            client_secret,
            access_token=x_token,
            user_ip=user_ip,
            **kwargs
        )

    def issue_authorization_code(
        self,
        client_id,
        client_secret,
        headers=None,
        uid=None,
        by_uid=None,
        user_ip=None,
        require_activation=None,
        **kwargs
    ):
        data = dict(
            client_id=client_id,
            client_secret=client_secret,
            consumer=self.consumer,
        )
        if uid:
            data.update(uid=str(uid))
        if by_uid is not None:
            data.update(by_uid=str(int(by_uid)))
        if user_ip is not None:
            data.update(user_ip=user_ip)
        if require_activation is not None:
            data.update(require_activation=str(int(require_activation)))
        data.update(remove_none_values(kwargs))

        params = self.params_from_user_info_data(data)

        headers = dict() if headers is None else dict(headers)

        # Заменяем заголовок Host и не пробрасываем TVM-тикет в OAuth
        oauth_host = urlparse(self.url).hostname
        headers['Host'] = oauth_host
        headers.pop('X-Ya-Service-Ticket', None)

        return self._request_with_retries_simple(
            bundle_api_error_detector,
            self.parse_json,
            url_suffix='/api/1/authorization_code/issue',
            method='POST',
            data=data,
            params=params,
            headers=headers,
            http_error_handler=oauth_error_handler,
        )

    def device_status(self, uid, **device_info):
        return self._request_with_retries_simple(
            bundle_api_error_detector,
            self.parse_json,
            url_suffix='/api/1/device/status',
            method='GET',
            params=dict(
                uid=uid,
                consumer=self.consumer,
                **device_info
            ),
            http_error_handler=oauth_error_handler,
        )

    def revoke_device(self, uid, device_id, user_ip=None):
        headers = dict()
        if user_ip is not None:
            headers.update({'Ya-Consumer-Client-Ip': user_ip})
        return self._request_with_retries_simple(
            bundle_api_error_detector,
            self.parse_json,
            url_suffix='/api/1/device/revoke',
            method='POST',
            data=dict(
                uid=str(uid),
                device_id=device_id,
                consumer=self.consumer,
            ),
            headers=headers,
            http_error_handler=oauth_error_handler,
        )

    def issue_device_code(self, client_id, code_strength, client_bound, **device_info):
        return self._request_with_retries_simple(
            bundle_api_error_detector,
            self.parse_json,
            url_suffix='/api/1/device_code/issue',
            method='POST',
            params=dict(
                client_id=client_id,
                code_strength=code_strength,
                client_bound=str(client_bound),
                consumer=self.consumer,
                **device_info
            ),
            http_error_handler=oauth_error_handler,
        )

    def check_device_code(self, device_code):
        return self._request_with_retries_simple(
            bundle_api_error_detector,
            self.parse_json,
            url_suffix='/api/1/device_code/check',
            method='GET',
            params=dict(
                device_code=device_code,
                consumer=self.consumer,
            ),
            http_error_handler=oauth_error_handler,
        )

    def _device_authorize_submit_or_commit(self, submit_or_commit, language, code, client_id, client_host=None,
                                           uid=None, cookie=None, authorization=None,
                                           **device_info):
        params = {
            'language': language,
            'client_id': client_id,
            'consumer': self.consumer,
        }
        if uid is not None:
            params['uid'] = uid
        params.update(device_info)
        headers = dict()
        if cookie is not None:
            headers['Ya-Client-Cookie'] = cookie
        if authorization is not None:
            headers['Ya-Consumer-Authorization'] = authorization
        if client_host is not None:
            headers['Ya-Client-Host'] = client_host
        return self._request_with_retries_simple(
            bundle_api_error_detector,
            self.parse_json,
            url_suffix='/iface_api/1/device/authorize/{}'.format(submit_or_commit),
            method='POST',
            params=params,
            data={'code': code},
            headers=headers,
            http_error_handler=oauth_error_handler,
        )

    def device_authorize_submit(self, language, code, client_id, client_host=None, uid=None,
                                cookie=None, authorization=None,
                                **device_info):
        return self._device_authorize_submit_or_commit('submit', language, code, client_id, client_host,
                                                       uid, cookie, authorization,
                                                       **device_info)

    def device_authorize_commit(self, language, code, client_id, client_host=None, uid=None,
                                cookie=None, authorization=None,
                                **device_info):
        return self._device_authorize_submit_or_commit('commit', language, code, client_id, client_host,
                                                       uid, cookie, authorization,
                                                       **device_info)


def get_oauth():
    return OAuth()  # pragma: no cover


def get_fast_oauth():
    return OAuth(timeout=settings.FAST_OAUTH_TIMEOUT, retries=settings.FAST_OAUTH_RETRIES)  # pragma: no cover
