# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import (
    BaseBuilder,
    parser_trivial,
)
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.exceptions import BaseCoreError
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.octopus')


class BaseOctopusError(BaseCoreError):
    pass


class OctopusPermanentError(BaseOctopusError):
    """Пятисотка или иная непредвиденная ошибка"""


class OctopusTemporaryError(BaseOctopusError):
    """Временная ошибка Octopus - стоит поретраиться"""


class OctopusSessionNotFound(OctopusPermanentError):
    """Сессия не найдена - передан невалидный SessionID"""


class OctopusAuthError(OctopusPermanentError):
    """Невалидный токен"""


class OctopusInvalidResponse(OctopusPermanentError):
    """Ошибка в ответе"""


def octopus_http_error_handler(response):
    if response.status_code in (502, 503):
        raise OctopusTemporaryError(u'Request failed with code=%s; response=%s' % (
            response.status_code,
            trim_message(response.content.decode('utf-8')),
        ))
    if response.status_code == 401:
        raise OctopusAuthError(u'Request failed with code=%s; response=%s' % (
            response.status_code,
            trim_message(response.content.decode('utf-8')),
        ))
    if response.status_code != 200:
        raise OctopusPermanentError(u'Request failed with code=%s; response=%s' % (
            response.status_code,
            trim_message(response.content.decode('utf-8')),
        ))


def octopus_session_log_http_error_handler(response):
    if response.status_code == 404:
        raise OctopusSessionNotFound()

    octopus_http_error_handler(response)


def create_session_error_detector(response, raw_response):
    if not response:
        raise OctopusInvalidResponse(u'No session id in response')


def session_status_error_detector(response, raw_response):
    if 'status' not in response or response['status'] not in settings.TELEPHONY_OK_STATUSES:
        raise OctopusInvalidResponse(
            u'Octopus returned error status=%s; response=%s',
            response.get('status', ''),
            trim_message(raw_response.content.decode('utf-8')),
        )


class Octopus(BaseBuilder, JsonParserMixin):
    """
    API для работы с телефонными звонками:
    https://wiki.yandex-team.ru/telephony/telephony-platform-mvp-passport-api/
    """
    base_error_class = BaseOctopusError
    temporary_error_class = OctopusTemporaryError
    parser_error_class = OctopusPermanentError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, graphite_logger=None,
                 use_tvm=False, **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='octopus')
        super(Octopus, self).__init__(
            url=url or settings.OCTOPUS_URL,
            timeout=timeout or settings.OCTOPUS_TIMEOUT,
            retries=retries or settings.OCTOPUS_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias='octopus' if use_tvm else None,
            **kwargs
        )

    def _create_session(self, data, url_suffix):
        return self._request_with_retries_simple(
            error_detector=create_session_error_detector,
            parser=parser_trivial,
            url_suffix=url_suffix,
            method='POST',
            json_data=data,
            headers={
                'Authorization': settings.OCTOPUS_AUTH_TOKEN,
            },
            http_error_handler=octopus_http_error_handler,
        )

    def create_session(self, first_code, second_code, caller, callee, language):
        data = {
            'firstCode': int(first_code),
            'secondCode': int(second_code),
            'caller': caller,
            'callee': callee,
            'locale': language,
        }
        return self._create_session(data, '/v0/create-call')

    def create_flash_call_session(self, caller, callee):
        data = {
            'caller': caller,
            'callee': callee,
        }
        return self._create_session(data, '/v0/create-pinger-call')

    def get_session_log(self, call_session_id):
        return self._request_with_retries_simple(
            error_detector=session_status_error_detector,
            parser=self.parse_json,
            url_suffix='/v0/call-status',
            method='GET',
            params={
                'sessionID': call_session_id,
            },
            headers={
                'Authorization': settings.OCTOPUS_AUTH_TOKEN,
            },
            http_error_handler=octopus_session_log_http_error_handler,
        )


def get_octopus():
    return Octopus()  # pragma: no cover
