# -*- coding: utf-8 -*-
from __future__ import absolute_import

import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.builders.passport.exceptions import (
    BasePassportError,
    PassportAccountDisabledError,
    PassportAccountNotFoundError,
    PassportActionImpossible,
    PassportActionNotRequiredError,
    PassportInvalidAccountTypeError,
    PassportLoginNotavailableError,
    PassportNoSubscriptionError,
    PassportPermanentError,
    PassportPhoneOperationExistsError,
    PassportPhonishCompletionNotStartedError,
    PassportPhonishCompletionOtherStartedError,
    PassportReceiverAccountNotFoundError,
    PassportTemporaryError,
    PassportTooFrequentChangePasswordError,
    PassportTrackInvalidStateError,
    PassportTrackNotFoundError,
    PassportUserNotVerifiedError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.passport')


def to_bool(value):
    return int(bool(value))


def error_handler(response):
    if response.status_code >= 500:
        log.warning(
            u'Request failed with with response=%s code=%s',
            trim_message(response.content.decode('utf-8')),
            response.status_code,
        )
        if response.status_code == 503:
            raise PassportTemporaryError('Request failed')
        else:
            raise PassportPermanentError('Server error')


def old_api_error_detector(response, raw_response):
    if response.get('errors'):
        error_code = response['errors'][0].get('code')
        error_description = response['errors'][0].get('message', error_code)
        if error_code == 'NoSubscription':
            raise PassportNoSubscriptionError()
        else:
            log.warning(u'Request failed: %s', error_description)
            raise PassportPermanentError(error_description)


def bundle_api_error_detector(response, raw_response):
    if response.get('errors'):
        error = response['errors'][0]
        if error in (
            'backend.blackbox_failed',
            'backend.database_failed',
            'backend.redis_failed',
            'backend.music_failed',
            'backend.billing_failed',
            'backend.bot_api_failed',
            'internal.temporary',
        ):
            raise PassportTemporaryError()
        elif error == 'account.invalid_type':
            raise PassportInvalidAccountTypeError()
        elif error == 'password.too_frequent_change':
            raise PassportTooFrequentChangePasswordError()
        elif error == 'action.not_required':
            raise PassportActionNotRequiredError()
        elif error == 'account.not_found':
            raise PassportAccountNotFoundError()
        elif error == 'login.notavailable':
            raise PassportLoginNotavailableError()
        elif error == 'phonish_completion.not_started':
            raise PassportPhonishCompletionNotStartedError()
        elif error == 'phonish_completion.other_started':
            raise PassportPhonishCompletionOtherStartedError()
        elif error == 'receiver_account.not_found':
            raise PassportReceiverAccountNotFoundError()
        elif error == 'operation.exists':
            raise PassportPhoneOperationExistsError()
        elif error == 'track.not_found':
            raise PassportTrackNotFoundError()
        elif error == 'track.invalid_state':
            raise PassportTrackInvalidStateError()
        elif error == 'account.disabled':
            raise PassportAccountDisabledError()
        elif error == 'user.not_verified':
            raise PassportUserNotVerifiedError()
        elif error == 'action.impossible':
            raise PassportActionImpossible()
        else:
            log.warning(u'Request failed: %s', error)
            raise PassportPermanentError(error)


class Passport(BaseBuilder, JsonParserMixin):

    base_error_class = BasePassportError
    temporary_error_class = PassportTemporaryError
    parser_error_class = PassportPermanentError

    def __init__(self, url=None, consumer=None, useragent=None, timeout=None, retries=None, graphite_logger=None,
                 use_tvm=False, **kwargs):
        self.consumer = consumer or settings.PASSPORT_CONSUMER
        timeout = timeout or settings.PASSPORT_TIMEOUT
        graphite_logger = graphite_logger or GraphiteLogger(service='passport')
        super(Passport, self).__init__(
            url=url or settings.PASSPORT_URL,
            timeout=timeout,
            retries=retries or settings.PASSPORT_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias='passport' if use_tvm else None,
            **kwargs
        )

    def _base_request(self, url_suffix, http_method, error_detector, get_args=None, post_args=None, headers=None):
        get_args = get_args or {}
        get_args.update(consumer=self.consumer)
        return self._request_with_retries_simple(
            error_detector=error_detector,
            parser=self.parse_json,
            url_suffix=url_suffix,
            method=http_method,
            params=get_args,
            data=post_args,
            http_error_handler=error_handler,
            headers=headers,
        )

    def subscribe(self, uid, sid):
        return self._base_request(
            url_suffix='1/account/%s/subscription/%s/' % (uid, sid),
            http_method='POST',
            error_detector=old_api_error_detector,
        )

    def unsubscribe(self, uid, sid):
        return self._base_request(
            url_suffix='1/account/%s/subscription/%s/' % (uid, sid),
            http_method='DELETE',
            error_detector=old_api_error_detector,
        )

    def app_password_created_send_notifications(self, uid, app_type, user_ip, app_name=None):
        return self._base_request(
            url_suffix='1/bundle/account/app_passwords/create/send_notifications/',
            http_method='POST',
            post_args={
                'uid': uid,
                'app_type': app_type,
                'app_name': app_name,
            },
            headers={
                'Ya-Consumer-Client-Ip': user_ip,
            },
            error_detector=bundle_api_error_detector,
        )

    def oauth_client_edited_send_notifications(self, client_id, client_title, redirect_uris_changed, scopes_changed,
                                               user_ip, cookies, host, uid=None):
        params = {
            'client_id': client_id,
            'client_title': client_title,
            'redirect_uris_changed': int(redirect_uris_changed),
            'scopes_changed': int(scopes_changed),
        }
        if uid is not None:
            params.update(uid=uid)
        return self._base_request(
            url_suffix='1/bundle/oauth/client/edit/send_notifications/',
            http_method='POST',
            post_args=params,
            headers={
                'Ya-Consumer-Client-Ip': user_ip,
                'Ya-Client-Cookie': cookies,
                'Ya-Client-Host': host,
            },
            error_detector=bundle_api_error_detector,
        )

    def send_challenge_email(self, uid, device_name, user_ip, user_agent, is_challenged):
        return self._base_request(
            url_suffix='1/bundle/auth/password/challenge/send_email/',
            http_method='POST',
            post_args={
                'uid': uid,
                'device_name': device_name,
                'is_challenged': to_bool(is_challenged),
            },
            headers={
                'Ya-Consumer-Client-Ip': user_ip,
                'Ya-Client-User-Agent': user_agent,
            },
            error_detector=bundle_api_error_detector,
        )

    def password_options(self, uid, is_changing_required=None, max_change_frequency_in_days=None,
                         notify_by_sms=None, show_2fa_promo=None, update_timestamp=None,
                         global_logout=None, comment=None, admin='robot:historydb'):
        params = {}

        if is_changing_required is not None:
            params['is_changing_required'] = to_bool(is_changing_required)

        if max_change_frequency_in_days is not None:
            params['max_change_frequency_in_days'] = max_change_frequency_in_days

        if notify_by_sms is not None:
            params['notify_by_sms'] = to_bool(notify_by_sms)

        if show_2fa_promo is not None:
            params['show_2fa_promo'] = to_bool(show_2fa_promo)

        if update_timestamp is not None:
            params['update_timestamp'] = update_timestamp

        if global_logout is not None:
            params['global_logout'] = to_bool(global_logout)

        if comment is not None and admin is not None:
            params['comment'] = comment
            params['admin_name'] = admin

        return self._base_request(
            url_suffix='2/account/%s/password_options/' % uid,
            http_method='POST',
            error_detector=bundle_api_error_detector,
            post_args=params,
        )

    def create_restoration_link(self, uid, link_type, ip, host, admin, comment=None, error_detector=None):
        params = {
            'uid': uid,
            'link_type': link_type,
            'admin_name': admin,
        }
        if comment is not None:
            params['comment'] = comment
        headers = {
            'Ya-Consumer-Client-Ip': ip,
            'Ya-Client-Host': host,
        }
        return self._base_request(
            url_suffix='1/bundle/restore/create_link/',
            http_method='POST',
            post_args=params,
            error_detector=error_detector,
            headers=headers,
        )

    def finish_phonish_completion(self, uid, completion_id, version, completion_args):
        post_args = {
            'uid': str(uid),
            'completion_id': completion_id,
            'version': version,
            'completion_args': completion_args,
        }
        return self._base_request(
            url_suffix='1/bundle/complete/finish_phonish_completion/',
            http_method='POST',
            post_args=post_args,
            error_detector=bundle_api_error_detector,
        )

    def finish_phonish_migration(self, phonish_uid, receiver_uid, migration_id,
                                 version, migration_args):
        post_args = {
            'phonish_uid': str(phonish_uid),
            'receiver_uid': str(receiver_uid),
            'migration_id': migration_id,
            'version': version,
            'migration_args': migration_args,
        }
        return self._base_request(
            url_suffix='1/bundle/complete/finish_phonish_migration/',
            http_method='POST',
            post_args=post_args,
            error_detector=bundle_api_error_detector,
        )

    def rfc_2fa_enable(self, login):
        return self._base_request(
            url_suffix='1/bundle/rfc_otp/enable/',
            http_method='POST',
            post_args={
                'login': login,
            },
            error_detector=bundle_api_error_detector,
        )

    def rfc_2fa_disable(self, login):
        return self._base_request(
            url_suffix='1/bundle/rfc_otp/disable/',
            http_method='POST',
            post_args={
                'login': login,
            },
            error_detector=bundle_api_error_detector,
        )

    def rfc_2fa_set_time(self, uid, totp_check_time):
        return self._base_request(
            url_suffix='1/bundle/rfc_otp/set_check_time/',
            http_method='POST',
            post_args={
                'uid': uid,
                'totp_check_time': totp_check_time,
            },
            error_detector=bundle_api_error_detector,
        )

    def bind_phone_from_phonish_to_portal(self, portal_uid, phonish_uid, user_ip):
        return self._base_request(
            url_suffix='2/bundle/phone/bind_phone_from_phonish_to_portal/',
            http_method='POST',
            post_args={
                'portal_uid': str(portal_uid),
                'phonish_uid': str(phonish_uid),
            },
            headers={
                'Ya-Consumer-Client-Ip': user_ip,
            },
            error_detector=bundle_api_error_detector,
        )

    def takeout_start_extract(self, uid):
        return self._base_request(
            url_suffix='1/bundle/takeout/extract/start/',
            http_method='POST',
            post_args={
                'uid': uid,
            },
            error_detector=bundle_api_error_detector,
        )

    def takeout_finish_extract(self, uid, archive_s3_key, archive_password=None):
        post_args = {
            'uid': uid,
            'archive_s3_key': archive_s3_key,
        }
        if archive_password:
            post_args.update(archive_password=archive_password)
        return self._base_request(
            url_suffix='1/bundle/takeout/extract/finish/',
            http_method='POST',
            post_args=post_args,
            error_detector=bundle_api_error_detector,
        )

    def takeout_delete_extract_result(self, uid):
        return self._base_request(
            url_suffix='1/bundle/takeout/extract/delete_result/',
            http_method='POST',
            post_args={
                'uid': uid,
            },
            error_detector=bundle_api_error_detector,
        )

    def reset_avatar(self, uid, avatar_key):
        post_args = {
            'uid': uid,
            'avatar_key': avatar_key,
        }
        return self._base_request(
            url_suffix='1/bundle/account/reset/avatar/',
            http_method='POST',
            post_args=post_args,
            error_detector=bundle_api_error_detector,
        )

    def set_default_avatar(self, uid, key, user_ip):
        post_args = {
            'uid': uid,
            'key': key,
        }
        return self._base_request(
            url_suffix='1/account/avatars/default/',
            http_method='POST',
            post_args=post_args,
            error_detector=bundle_api_error_detector,
            headers={
                'Ya-Consumer-Client-Ip': user_ip,
            },
        )

    def reset_display_name(self, uid, full_name=None, public_name=None):
        assert bool(full_name) != bool(public_name)
        post_args = {
            'uid': uid,
            'full_name': full_name,
            'public_name': public_name,
        }
        post_args = {k: post_args[k] for k in post_args if post_args[k]}
        return self._base_request(
            url_suffix='1/bundle/account/reset/display_name/',
            http_method='POST',
            post_args=post_args,
            error_detector=bundle_api_error_detector,
        )

    def account_options(
        self,
        uid,
        comment=None,
        admin=None,
        takeout_subscription=None,
        takeout_delete_subscription=None,
        sms_2fa_on=None,
        forbid_disabling_sms_2fa=None,
    ):
        params = {}

        if takeout_subscription is not None:
            params['takeout_subscription'] = to_bool(takeout_subscription)
        if takeout_delete_subscription is not None:
            params['takeout_delete_subscription'] = to_bool(
                takeout_delete_subscription,
            )
        if sms_2fa_on is not None:
            params['sms_2fa_on'] = to_bool(sms_2fa_on)
        if forbid_disabling_sms_2fa is not None:
            params['forbid_disabling_sms_2fa'] = to_bool(forbid_disabling_sms_2fa)

        if comment is not None and admin is not None:
            params['comment'] = comment
            params['admin_name'] = admin

        return self._base_request(
            url_suffix='2/account/%s/options/' % uid,
            http_method='POST',
            error_detector=bundle_api_error_detector,
            post_args=params,
        )

    def get_phonish_uid_by_phone(self, track_id, user_ip, use_device_id=False):
        return self._base_request(
            url_suffix='/1/bundle/account/phonish/uid_by_phone/',
            http_method='GET',
            error_detector=bundle_api_error_detector,
            get_args={'track_id': track_id, 'use_device_id': use_device_id},
            headers={'Ya-Consumer-Client-Ip': user_ip},
        )

    def send_account_modification_notifications(
        self,
        host,
        user_ip,
        uid,
        event_name,
        mail_enabled=True,
        push_enabled=True,
        social_provider=None,
    ):
        post_args = dict(
            uid=uid,
            event_name=event_name,
            mail_enabled='1' if mail_enabled else '0',
            push_enabled='1' if push_enabled else '0',
        )
        if social_provider:
            post_args.update(social_provider=social_provider)
        return self._base_request(
            url_suffix='/1/bundle/account/modification/send_notifications/',
            http_method='POST',
            error_detector=bundle_api_error_detector,
            post_args=post_args,
            headers={
                'Ya-Consumer-Client-Ip': user_ip,
                'Ya-Client-Host': host,
            },
        )


def get_passport():
    return Passport()  # pragma: no cover
