# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.builders.perimeter_api.exceptions import (
    BasePerimeterApiError,
    PerimeterApiPermanentError,
    PerimeterApiTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.perimeter')


def perimeter_error_handler(response):
    if response.status_code >= 500:
        raise PerimeterApiPermanentError(
            'Server error (%s): %s' % (response.status_code, trim_message(response.content.decode('utf-8'))),
        )


def error_detector(response, _raw_response):
    if response.get('status') == 'error':
        error_msg = response.get('error')
        if error_msg in ['database.failed', 'passport.failed']:
            raise PerimeterApiTemporaryError(error_msg)
        else:
            raise PerimeterApiPermanentError(error_msg)


def recreate_totp_secret_error_detector(response, raw_response):
    error_detector(response, raw_response)
    if not response.get('secret'):
        raise PerimeterApiPermanentError('Absent or empty secret field.')


class PerimeterApi(BaseBuilder, JsonParserMixin):
    base_error_class = BasePerimeterApiError
    temporary_error_class = PerimeterApiTemporaryError
    parser_error_class = PerimeterApiPermanentError

    def __init__(self, perimeter=None, useragent=None, timeout=None, retries=None, graphite_logger=None,
                 client_side_cert=None, client_side_key=None):
        timeout = timeout or settings.PERIMETER_API_TIMEOUT
        graphite_logger = graphite_logger or GraphiteLogger(service='perimeter')

        super(PerimeterApi, self).__init__(
            url=perimeter or settings.PERIMETER_API_URL,
            timeout=timeout,
            retries=retries or settings.PERIMETER_API_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            client_side_cert=client_side_cert or settings.PERIMETER_API_CLIENT_CERT,
            client_side_key=client_side_key or settings.PERIMETER_API_CLIENT_CERT_KEY,
        )

    def recreate_totp_secret(self, login):
        return self._request_with_retries_simple(
            recreate_totp_secret_error_detector,
            self.parse_json,
            url_suffix='recreate-totp-secret/{}/'.format(login),
            http_error_handler=perimeter_error_handler,
        )


def get_perimeter_api():
    return PerimeterApi()  # pragma: no cover
