# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.builders.phone_squatter.exceptions import (
    BasePhoneSquatterError,
    PhoneSquatterInvalidResponseError,
    PhoneSquatterPermanentError,
    PhoneSquatterPhoneNumberNotTrackedError,
    PhoneSquatterPhoneNumberUntrackableError,
    PhoneSquatterTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.phone_squatter')


def error_handler(response):
    if response.status_code == 422:
        raise PhoneSquatterPhoneNumberUntrackableError()
    elif response.status_code == 402:
        raise PhoneSquatterPhoneNumberNotTrackedError()
    elif response.status_code not in (200, 202):
        raise PhoneSquatterPermanentError(
            u'Request failed with with status=%s response=`%s`' % (
                response.status_code,
                trim_message(response.content.decode('utf-8')),
            ),
        )


class PhoneSquatter(BaseBuilder, JsonParserMixin):
    """
    https://wiki.yandex-team.ru/passport/phonesquatter
    """
    base_error_class = BasePhoneSquatterError
    temporary_error_class = PhoneSquatterTemporaryError
    parser_error_class = PhoneSquatterInvalidResponseError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, graphite_logger=None):
        super(PhoneSquatter, self).__init__(
            url=url or settings.PHONE_SQUATTER_URL,
            timeout=timeout or settings.PHONE_SQUATTER_TIMEOUT,
            retries=retries or settings.PHONE_SQUATTER_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger or GraphiteLogger(service='phone_squatter'),
            tvm_dst_alias='phone_squatter',
        )

    def _make_request(self, url_suffix, method, query_params, json_data):
        return self._request_with_retries_simple(
            url_suffix=url_suffix,
            method=method,
            params=query_params,
            json_data=json_data,
            http_error_handler=error_handler,
            parser=self.parse_json,
            error_detector=None,
        )

    def get_change_status(self, phone_number, request_id, allow_cached=False):
        return self._make_request(
            url_suffix='/phone/change/status',
            method='POST',
            query_params={
                'reqid': request_id,
                'allow_cached': 'true' if allow_cached else 'false',
            },
            json_data={'phone': phone_number},
        )

    def start_tracking(self, phone_number, request_id, is_draft=False):
        return self._make_request(
            url_suffix='/phone/tracking/start',
            method='POST',
            query_params={
                'reqid': request_id,
                'draft': 'true' if is_draft else 'false',
            },
            json_data={'phone': phone_number},
        )


def get_phone_squatter():
    return PhoneSquatter()  # pragma: no cover
