# -*- coding: utf-8 -*-
from collections import namedtuple
import json
import logging

from passport.backend.core.builders.base.base import (
    BaseBuilder,
    parser_trivial,
)
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.builders.push_api.exceptions import (
    BasePushApiError,
    PushApiInvalidRequestError,
    PushApiInvalidResponseError,
    PushApiTemporaryError,
    PushApiUnsupportedPlatformError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.utils.common import noneless_dict
from passport.backend.utils.string import smart_bytes


log = logging.getLogger('passport.push_api')

SUPPORTED_PLATFORMS = {
    'android': 'gcm',
    'apns': 'apns',
    'apple': 'apns',
    'iphone': 'apns',
    'ipad': 'apns',
    'gcm': 'gcm',
}


Subscription = namedtuple(
    'Subscription',
    'id, client, session, app, uuid, platform, device, filter, ttl, '
    'init_time, last_sent, raw_extra, extra',
)


SUBSCRIPTION_EXTRA_FIELDS = ['am_version', 'login_id', 'app_version']
SUBSCRIPTION_EXTRA_REVERSE_MAP = {k: i for i, k in enumerate(SUBSCRIPTION_EXTRA_FIELDS)}


SubscriptionExtra = namedtuple('SubscriptionExtra', SUBSCRIPTION_EXTRA_FIELDS)


def make_extra_data(**extra_fields):
    return json.dumps({
        SUBSCRIPTION_EXTRA_REVERSE_MAP[k]: v for k, v in extra_fields.items()
        if v is not None
    })


def http_detect_errors(raw_response):
    if raw_response.status_code != 200:
        message = u'Request failed with response={} code={}'.format(
            trim_message(raw_response.content.decode('utf-8')),
            raw_response.status_code,
        )
        log.warning(message)
        raise PushApiInvalidResponseError(message)


def detect_errors_list_response(data, raw_response):
    if not isinstance(data, list):
        message = u'Unexpected response format response={}'.format(
            trim_message(raw_response.content.decode('utf-8')),
        )
        log.warning(message)
        raise PushApiInvalidResponseError(message)


def parse_subscription_extra(raw_extra_data):
    if raw_extra_data:
        try:
            extra_data = json.loads(raw_extra_data)
            if not isinstance(extra_data, dict):
                raise ValueError('{}, not a dict'.format(type(extra_data)))
        except ValueError as err:
            log.warning(
                'Unsupported extra data format ({}) {}'.format(
                    err, trim_message(raw_extra_data)
                ),
            )
            extra_data = {}
    else:
        extra_data = {}

    return SubscriptionExtra(**{
        field: extra_data.get(str(i))
        for i, field in enumerate(SUBSCRIPTION_EXTRA_FIELDS)
    })


def subscription_dict_to_obj(raw_sub):
    return Subscription(
        id=raw_sub['id'],
        client=raw_sub.get('client'),
        session=raw_sub.get('session'),
        app=raw_sub.get('app'),
        uuid=raw_sub.get('uuid'),
        platform=raw_sub.get('platform'),
        device=raw_sub.get('device'),
        filter=raw_sub.get('filter'),
        ttl=raw_sub.get('ttl'),
        init_time=raw_sub.get('init_time'),
        last_sent=raw_sub.get('last_sent'),
        raw_extra=raw_sub.get('extra'),
        extra=parse_subscription_extra(raw_sub.get('extra')),
    )


class PushApi(BaseBuilder, JsonParserMixin):
    base_error_class = BasePushApiError
    temporary_error_class = PushApiTemporaryError
    parser_error_class = PushApiInvalidResponseError
    accept_empty_response = True

    def __init__(
        self, url=None, useragent=None, timeout=None, retries=None,
        graphite_logger=None, use_tvm=True, tvm_dst_alias='push_api',
    ):
        timeout = timeout or settings.PUSH_API_TIMEOUT
        retries = retries or settings.PUSH_API_RETRIES
        graphite_logger = graphite_logger or GraphiteLogger(service='push')
        super(PushApi, self).__init__(
            graphite_logger=graphite_logger,
            logger=log,
            retries=retries,
            timeout=timeout,
            url=url or settings.PUSH_API_URL,
            useragent=useragent,
            tvm_dst_alias=tvm_dst_alias if use_tvm else None,
        )
        self.use_tvm = use_tvm

    def translate_platform(self, platform):
        platform = platform.lower()
        for supported_platform in SUPPORTED_PLATFORMS.keys():
            if platform.startswith(supported_platform):
                return SUPPORTED_PLATFORMS[supported_platform]
        raise PushApiUnsupportedPlatformError()

    def error_detector(self, data, raw_response):
        if raw_response.status_code != 200 or raw_response.content != b'OK':
            log.warning(
                u'Request failed with response=%s code=%s',
                trim_message(raw_response.content.decode('utf-8')),
                raw_response.status_code,
            )
            raise PushApiInvalidRequestError()

    def sync_error_detector(self, data, raw_response):
        if raw_response.status_code != 200:
            log.warning(
                u'Sync send request failed with response=%s code=%s',
                trim_message(raw_response.content.decode('utf-8')),
                raw_response.status_code,
            )
            raise PushApiInvalidRequestError()

    def parse_send_sync_result(self, raw_response):
        """
        Документация: https://console.push.yandex-team.ru/#api-reference-batch-send
        (раздел "пример отчёта" и далее)
        """
        data = json.loads(raw_response.content)
        log.debug('Push api response: {}'.format(data))
        try:
            if not isinstance(data, list):
                if not isinstance(data, dict):
                    raise ValueError('List or dict expected')
                elif 'id' in data:
                    # Если подписка была одна, ксива отвечает не массивом
                    return {data['id']: data}
                if data.get('code') in (202, 204, 205):  # Подписок не найдено или подписка отклонена
                    return {}
                else:
                    raise ValueError('Unknown send response {}'.format(data))
            return {r['id']: r for r in data}
        except (ValueError, TypeError, KeyError) as err:
            message = u'Wrong push api response={} code={} ({})'.format(
                trim_message(raw_response.content.decode('utf-8')),
                raw_response.status_code,
                err,
            )
            log.warning(message)
            raise PushApiInvalidResponseError(message)

    def register(self, app_name, cert, platform):
        """ Зарегистрировать приложение в сервисе Push API """
        return self._request_with_retries_simple(
            data=smart_bytes(cert),
            error_detector=self.error_detector,
            method='POST',
            params=dict(
                app_name=app_name,
                service=settings.PUSH_API_SERVICE_NAME,
            ),
            parser=parser_trivial,
            url_suffix='v2/register/app/%s' % self.translate_platform(platform),
        )

    def send(self, event, payload, user, ttl=None, sync=False):
        """ Отправить пользователю пуш """
        request_kwargs = dict(
            json_data=payload,
            error_detector=self.error_detector,
            method='POST',
            params=noneless_dict(
                event=event,
                user=user,
                ttl=ttl,
                service=settings.PUSH_API_SERVICE_NAME,
            ),
            parser=parser_trivial,
            url_suffix='v2/send',
            headers={
                'X-DeliveryMode': 'queued',
            },
        )
        if sync:
            request_kwargs.update(
                error_detector=self.sync_error_detector,
                parser=self.parse_send_sync_result,
            )
            request_kwargs['headers']['X-DeliveryMode'] = 'direct'
        return self._request_with_retries_simple(**request_kwargs)

    def subscribe(
        self, app_name, device_token, platform, user, uuid,
        client=None, device=None, extra_fields=None, extra=None, filter_=None,
    ):
        """ Подписать инсталяцию приложения на пуши """
        if extra_fields:
            extra = make_extra_data(**extra_fields)

        return self._request_with_retries_simple(
            data=noneless_dict(
                push_token=device_token,
                filter_=filter_,
                extra=extra,
            ),
            error_detector=self.error_detector,
            method='POST',
            params=noneless_dict(
                app_name=app_name,
                client=client,
                device=device,
                platform=self.translate_platform(platform),
                service=settings.PUSH_API_SERVICE_NAME,
                user=user,
                uuid=uuid,
            ),
            parser=parser_trivial,
            url_suffix='v2/subscribe/app',
        )

    def unsubscribe(self, user, uuid):
        """ Отписать инсталяцию приложения от пушей """
        return self._request_with_retries_simple(
            error_detector=self.error_detector,
            method='POST',
            params=dict(
                service=settings.PUSH_API_SERVICE_NAME,
                user=user,
                uuid=uuid,
            ),
            parser=parser_trivial,
            url_suffix='v2/unsubscribe/app',
        )

    def parse_list_result(self, raw_response):
        result_list = self.parse_json(raw_response)
        if not isinstance(result_list, list):
            raise self.parser_error_class(
                'Wrong list result format ({}, not a list)'.format(type(result_list)),
            )
        try:
            return [
                subscription_dict_to_obj(raw_sub)
                for raw_sub in result_list
            ]
        except KeyError as err:
            raise self.parser_error_class('Missing list result key: {}'.format(err))

    def list(self, user):
        """
        Получить список подписок пользователя
        https://console.push.yandex-team.ru/#api-reference-list

        Возвращает namedtuple со следующими атрибутами:
        id: (str) id подписки
        client: (str, optional) наименование типа клиента (из вызова subscribe)
        session: (str) id сессии
        app: (str) наименование программы, например 'ru.yandex.mail'
        uuid: (str, optional) uuid подписки (из вызова subscribe)
        platform: (str) тип платформы, например 'gcm'
        device: (str, optional) device id (из вызова subscribe)
        filter: (str, optional) список фильтров
        ttl (int): ttl записи в часах
        extra (SubscriptionExtra): разобранное extra-поле из вызова subscribe
        raw_extra (str, optional): сырое значение extra-поля
        init_time (int): unix timestamp последнего обновления подписки
        last_sent (int): unix timestamp последней отправки по подписке
        """
        return self._request_with_retries_simple(
            error_detector=detect_errors_list_response,
            http_error_handler=http_detect_errors,
            method='GET',
            params=dict(
                service=settings.PUSH_API_SERVICE_NAME,
                user=user,
            ),
            parser=self.parse_list_result,
            url_suffix='v2/list',
        )


def get_push_api(timeout=None):
    return PushApi(timeout=timeout)  # pragma: no cover
