# -*- coding: utf-8 -*-
import json
import logging

from passport.backend.core.builders.base.base import (
    BaseBuilder,
    parser_trivial,
)
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.builders.sender_api.exceptions import (
    BaseSenderApiError,
    SenderApiInvalidRequestError,
    SenderApiInvalidResponseError,
    SenderApiTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.utils.string import smart_unicode


log = logging.getLogger('passport.sender_api')


class SenderApi(BaseBuilder, JsonParserMixin):
    base_error_class = BaseSenderApiError
    temporary_error_class = SenderApiTemporaryError
    parser_error_class = SenderApiInvalidResponseError

    def __init__(
        self, url=None, useragent=None, timeout=None, retries=None,
        graphite_logger=None, tvm_dst_alias='sender_api',
    ):
        timeout = timeout or settings.SENDER_API_TIMEOUT
        retries = retries or settings.SENDER_API_RETRIES
        graphite_logger = graphite_logger or GraphiteLogger(service='sender')
        super(SenderApi, self).__init__(
            graphite_logger=graphite_logger,
            logger=log,
            retries=retries,
            timeout=timeout,
            url=url or settings.SENDER_API_URL,
            useragent=useragent,
            tvm_dst_alias=tvm_dst_alias,
        )

    def _get_error_message(self, raw_response):
        try:
            data = self.parse_json(raw_response)
            message = smart_unicode(data.get('result', {}).get('error') or data)
        except self.parser_error_class:
            message = smart_unicode(raw_response.content)
        return u'Sender api returned error code={}: {}'.format(
            raw_response.status_code, trim_message(message),
        )

    def http_error_handler(self, raw_response):
        if 400 <= raw_response.status_code <= 599:
            message = self._get_error_message(raw_response)
            log.warning(message)
            if raw_response.status_code >= 500:
                error_class = SenderApiTemporaryError
            else:
                error_class = SenderApiInvalidRequestError
            raise error_class(message)
        elif raw_response.status_code != 200:
            raise SenderApiInvalidResponseError(
                'Unexpected sender api response code {} data={}'.format(
                    raw_response.status_code,
                    trim_message(parser_trivial(raw_response)),
                ),
            )

    def set_unsubscriptions(self, email, unsubscribe_list, subscribe_list):
        """
        Установить состояние отписок
        https://github.yandex-team.ru/sendr/sendr/blob/master/docs/unsubscribe-api.md#изменение-стейта
        Состояние будет изменено только для переданных list_id

        :param email: адрес
        :type email: str
        :param unsubscribe_list: список list_id на отписку
        :type unsubscribe_list: List[int]
        :param subscribe_list: список list_id на удаление из отписки
        :type subscribe_list: List[int]
        :return: обработанный JSON ответа
        """
        states = [
            {'list_id': list_id, 'unsubscribed': True} for
            list_id in unsubscribe_list
        ] + [
            {'list_id': list_id, 'unsubscribed': False} for
            list_id in subscribe_list
        ]
        return self._request_with_retries_simple(
            data=dict(
                state=json.dumps(states),
            ),
            error_detector=None,
            http_error_handler=self.http_error_handler,
            method='POST',
            params=dict(
                email=email,
            ),
            parser=self.parse_json,
            url_suffix='api/0/{}/external/unsubscribe/set'.format(
                settings.SENDER_API_UNSUBSCRIBE_EXT_ACCOUNT,
            ),
        )

    def copy_unsubscriptions(self, email_src, email_dst):
        """
        Перенос отписок
        https://github.yandex-team.ru/sendr/sendr/blob/master/docs/unsubscribe-api.md#перенос-отписок-между-имейлами
        Все отписки, которые есть на email_src, появятся на email_dst

        :param email_src: адрес-источник
        :type email_src: str
        :param email_dst: адрес-источник
        :type email_dst: str
        :return: обработанный JSON ответа
        """
        return self._request_with_retries_simple(
            error_detector=None,
            http_error_handler=self.http_error_handler,
            method='POST',
            params=dict(
                src=email_src,
                dst=email_dst,
            ),
            parser=self.parse_json,
            url_suffix='api/0/{}/external/unsubscribe/copy'.format(
                settings.SENDER_API_UNSUBSCRIBE_EXT_ACCOUNT,
            ),
        )
