# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.builders.shakur.exceptions import (
    BaseShakurError,
    ShakurAuthorizationInvalidError,
    ShakurPermanentError,
    ShakurTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.shakur')


def shakur_error_detector(response, _raw_response):
    if _raw_response.status_code == 200:
        if 'found' not in response:
            raise ShakurPermanentError("No 'found' attribute in response")
        if 'passwords' not in response:
            raise ShakurPermanentError("No 'passwords' attribute in response")
        if not all(item.get('sha1') for item in response['passwords']):
            raise ShakurPermanentError("Not all passwords have 'sha1' attribute in response")


def shakur_http_error_handler(raw_response):
    if raw_response.status_code >= 500:
        log.warning(
            u'Request failed with response=%s code=%s',
            trim_message(raw_response.content.decode('utf-8')),
            raw_response.status_code,
        )
        raise ShakurTemporaryError('Server is down')
    elif raw_response.status_code == 403:
        raise ShakurAuthorizationInvalidError('Invalid tvm ticket')
    elif raw_response.status_code != 200:
        raise ShakurPermanentError('Bad response status code: %s' % raw_response.status_code)


class Shakur(BaseBuilder, JsonParserMixin):
    base_error_class = BaseShakurError
    temporary_error_class = ShakurTemporaryError
    parser_error_class = ShakurPermanentError

    """
    Дока к апи: https://wiki.yandex-team.ru/ps/documentation/shakur/
    """

    def __init__(self, url=None, timeout=None, retries=None, graphite_logger=None, use_tvm=False, useragent=None,
                 **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='shakur')
        use_tvm = use_tvm or settings.SHAKUR_USE_TVM
        super(Shakur, self).__init__(
            url=url or settings.SHAKUR_URL,
            timeout=timeout or settings.SHAKUR_TIMEOUT,
            retries=retries or settings.SHAKUR_RETRIES,
            logger=log,
            graphite_logger=graphite_logger,
            tvm_dst_alias='shakur' if use_tvm else None,
            useragent=useragent,
            **kwargs
        )

    def _make_request(self, method='GET', url_suffix='', params=None, data=None, parser=None, **kwargs):
        return self._request_with_retries_simple(
            url_suffix=url_suffix,
            method=method,
            params=params,
            data=data,
            parser=parser or self.parse_json,
            http_error_handler=shakur_http_error_handler,
            error_detector=shakur_error_detector,
            **kwargs
        )

    def check_password(self, sha_prefix):
        params = {
            'sha1': sha_prefix,
            'length': settings.SHAKUR_LIMIT,
        }

        return self._make_request(
            url_suffix='api/shakur/checkpassword',
            method='GET',
            params=params,
        )


def get_shakur():
    return Shakur()  # pragma: no cover
