# -*- coding: utf-8 -*-

import re

from passport.backend.core.builders.base.faker.fake_builder import BaseFakeBuilder
from passport.backend.core.builders.social_api import SocialApi
from passport.backend.core.test.consts import TEST_SOCIAL_TASK_ID1
from passport.backend.core.test.test_utils import traverse_dict_from_leaves_to_root
from passport.backend.core.undefined import Undefined
from six import iteritems
from six.moves.urllib.parse import urlparse


_url_path_to_method_name = {
    ('GET', r'^/api/user/(\d+)/profile$'): 'get_profiles_by_uid',
    ('DELETE', r'^/api/profile/(\d+)$'): 'delete_profile',
    ('POST', r'^/api/profile/(\d+)$'): 'set_authentificate_profile',
    ('DELETE', r'^/api/profile/(\d+)/subscription/(\d+)$'): 'delete_subscription',
    ('PUT', r'^/api/profile/(\d+)/subscription/(\d+)$'): 'create_subscription',
    ('GET', r'^/api/task/[0-9a-z_-]+$'): 'get_task_data',
    ('DELETE', r'^/api/task/[0-9a-z_-]+$'): 'delete_task',
    ('GET', r'^/api/profiles$'): 'get_profiles',
    ('POST', r'^/api/task/[0-9a-z_-]+/bind$'): 'bind_task_profile',
    ('POST', r'^/api/delete_social_data$'): 'delete_social_data',
    ('DELETE', r'^/api/user/(\d+)'): 'delete_all_profiles_by_uid',
    ('POST', r'^/api/token/delete_from_account$'): 'delete_tokens_from_account',
}

EXISTING_TASK_ID = TEST_SOCIAL_TASK_ID1
EXISTING_USERID = '57575757575'
PROFILE_EMAIL = 'some-mail@example.com'
GOOGLE_AVATAR = {
    '0x0': 'https://lh3.googleusercontent.com/-XdUIqdskCWA/AAAAAAAAAAI/AAAAAAAAAAA/2252rscbv5M/photo.jpg',
}
GOOGLE_PROVIDER = {'code': 'gg', 'name': 'google', 'id': 5}
FACEBOOK_AVATAR = {
    '50x50': 'https://fbcdn-profile-a.akamaihd.net/hprofile-ak-xpf1/v/t1.0-1/p50x50/10968192_1544225485863438_'
             '5927421952288376322_n.jpg?oh=8bedf1aaba1fd210ce57997da2aa00e9&oe=555A6064&__gda__='
             '1431383307_6a23dd1f2a5aba629a00e45a7919b95a',
    '200x600': 'https://fbcdn-profile-a.akamaihd.net/hprofile-ak-xpf1/v/t1.0-1/p200x200/10968192_'
               '1544225485863438_5927421952288376322_n.jpg?oh=9e1784834bc1d9df0741a933e2f7a8cb&oe=556E5611&'
               '__gda__=1431026301_377c73e0cfadd6418fcb55f838bc16df',
    '100x300': 'https://fbcdn-profile-a.akamaihd.net/hprofile-ak-xpf1/v/t1.0-1/p100x100/10968192_1544225485863438_'
               '5927421952288376322_n.jpg?oh=54fe1029be0f3a37ced761ccd95e979c&oe=5559F132&__gda__='
               '1433147998_1dcc292ea6c0c307a9a173756dfec99e',
    '50x150': 'https://fbcdn-profile-a.akamaihd.net/hprofile-ak-xpf1/v/t1.0-1/p50x50/10968192_'
              '1544225485863438_5927421952288376322_n.jpg?oh=8bedf1aaba1fd210ce57997da2aa00e9&oe='
              '555A6064&__gda__=1431383307_6a23dd1f2a5aba629a00e45a7919b95a'
}
FIRSTNAME = 'Firstname'
LASTNAME = 'Lastname'
USERNAME = 'some.user'

FACEBOOK_BUSINESS_ID = 1
FACEBOOK_BUSINESS_TOKEN = 'abcdef1234567890'

PROVIDERS_MAP = {
    'vkontakte': {
        'name': 'vkontakte',
        'code': 'vk',
        'id': 1,
    },
    'facebook': {
        'name': 'facebook',
        'code': 'fb',
        'id': 2,
    },
    'twitter': {
        'name': 'twitter',
        'code': 'tw',
        'id': 3,
    },
    'mailru': {
        'name': 'mailru',
        'code': 'mr',
        'id': 4,
    },
    'google': {
        'name': 'google',
        'code': 'gg',
        'id': 5,
    },
    'odnoklassniki': {
        'name': 'odnoklassniki',
        'code': 'ok',
        'id': 6,
    },
    'telegram': {
        'name': 'telegram',
        'code': 'tg',
        'id': 7,
    },
    'lastfm': {
        'name': 'lastfm',
        'code': 'lf',
        'id': 8,
    },
    'instagram': {
        'name': 'instagram',
        'code': 'ig',
        'id': 9,
    },
}

APP_PARTY_YANDEX = 1
APP_PARTY_3RD_PARTY = 2


def get_delete_task_response():
    return {
        'status': 'ok',
        'deleted': True,
    }


def get_profiles_response(profiles=Undefined):
    if profiles is None:
        profiles = []
    if profiles is Undefined:
        profiles = [
            dict(uid=100000000, profile_id=100, allow_auth=True),
            dict(uid=100000001, profile_id=101, allow_auth=False),
            dict(uid=100000002, profile_id=102, allow_auth=True),
        ]
    profiles = [profile_item(**p) for p in profiles]
    return {'profiles': profiles}


def get_template_profile_full_response(uid, profile_id, allow_auth, sid, provider):
    return profile_item(
        uid=uid,
        profile_id=profile_id,
        allow_auth=allow_auth,
        provider=provider,
        provider_code=PROVIDERS_MAP[provider]['code'],
        person=get_persons(profile_id),
        tokens=[get_tokens(profile_id)],
        subscriptions=[sid] if sid is not None else [],
    )


def get_tokens(profile_id):
    return {
        'profile_id': profile_id,
        'application': 'facebook-share',
        'permissions': 'user_about_me user_birthday offline_access',
        'secret': 'mega-super-secret',
        'token_id': 45,
        'value': '149040791774107.9p26W6FlJRwwcrpnfYHbgg115qk',
        'expired': '2010-09-13 20:14:39',
        'expired_ts': 100500,
        'created': '2010-08-13 20:14:39',
        'created_ts': 100500,
        'confirmed': None,
        'confirmed_ts': None,
        'verified': '2011-06-25 01:56:21',
        'verified_ts': 100500,
    }


def get_persons(profile_id):
    return {
        'firstname': 'FirstName',
        'lastname': 'LastName',
        'profile_id': profile_id,
        'birthday': '1989-12-28',
        'gender': 'm',
        'nickname': '',
    }


def get_profiles_full_response():
    return {
        'profiles': [
            get_template_profile_full_response(100000000, 100, True, 25, 'google'),
            get_template_profile_full_response(100000001, 101, False, 83, 'vkontakte'),
            get_template_profile_full_response(100000002, 102, True, 2, 'facebook'),
            get_template_profile_full_response(100000003, 103, True, 74, 'twitter'),
            get_template_profile_full_response(100000004, 104, False, None, 'lastfm'),
            get_template_profile_full_response(100000005, 105, False, None, 'instagram'),
        ],
    }


def get_profiles_no_profiles():
    return {'profiles': []}


def get_ok_response():
    return {
        'status': 'ok',
    }


def get_bind_response():
    return {
        'status': 'ok',
        'uid': 100500,
        'profile_id': 123456789,
    }


def get_profile_info(uid, profile_id, auth_allow=None):
    response = {
        'profile': {
            'profile_id': profile_id,
            'addresses': [
                'http://www.facebook.com/profile.php?id=123',
                'http://www.facebook.com/facepalm',
            ],
            'provider': 'facebook',
            'uid': uid,
            'userid': '123',
            'username': 'facepalm',
        }
    }
    if auth_allow is not None:
        response['profile']['auth_allow'] = bool(auth_allow)
    return response


def get_subscription_info(profile_id, sid):
    return {
        'subscription': {
            'profile_id': profile_id,
            'sid': sid,
        }
    }


def profile_not_found_error():
    return {
        'error': {
            'description': 'Profile not found',
            'name': 'profile-not-found',
            'request_id': '7b240321-3',
        }
    }


def social_api_person_item(firstname=FIRSTNAME, lastname=LASTNAME, profile_id=123,
                           email=PROFILE_EMAIL, birthday='1989-12-28', gender='m',
                           nickname=''):
    assert gender in {'m', 'f'} or gender is None
    return {
        'firstname': firstname or '',
        'lastname': lastname or '',
        'profile_id': profile_id,
        'birthday': birthday or 'None',
        'gender': gender or '',
        'nickname': nickname or '',
        'email': email or '',
        'phone_number': '',
    }


def token_item(profile_id=123):
    return {
        'profile_id': profile_id,
        'application': 'facebook-share',
        'permissions': 'user_about_me user_birthday offline_access',
        'secret': 'mega-super-secret',
        'token_id': 26,
        'value': '149040791774107.9p26W6FlJRwwcrpnfYHbgg115qk',
        'expired': '2010-09-13 20:14:39',
        'expired_ts': 100500,
        'created': '2010-08-13 20:14:39',
        'created_ts': 100500,
        'confirmed': None,
        'confirmed_ts': None,
        'verified': '2011-06-25 01:56:21',
        'verified_ts': 100500,
    }


def subscription_not_found_error():
    return {
        'error': {
            'description': 'Subscription not found',
            'name': 'subscription-not-found',
            'request_id': '7b240321-3',
        }
    }


def profile_item(uid=1, username=USERNAME, profile_id=123, userid=EXISTING_USERID,
                 provider='facebook', provider_code='fb', allow_auth=True,
                 addresses=Undefined, person=None, tokens=None, subscriptions=None,
                 expand_provider=False):
    result = dict(
        username=username,
        uid=uid,
        profile_id=profile_id,
        userid=userid,
        allow_auth=allow_auth,
        provider=PROVIDERS_MAP[provider] if expand_provider else provider,
        provider_code=provider_code,
    )

    if addresses is Undefined:
        addresses = [
            'http://www.facebook.com/profile.php?id=%(userid)s',
            'http://www.facebook.com/some.user',
        ]
    result['addresses'] = addresses or []

    if person is not None:
        result['person'] = person
    if tokens is not None:
        result['tokens'] = tokens
    if subscriptions is not None:
        result['subscriptions'] = [dict(sid=sid) for sid in subscriptions]
    return result


def task_data_response(username=USERNAME, firstname=FIRSTNAME, lastname=LASTNAME,
                       avatar=Undefined, userid=EXISTING_USERID, provider_id=5, provider_code='gg',
                       provider_name='google', email=PROFILE_EMAIL, links=None,
                       task_id=EXISTING_TASK_ID, with_business=False,
                       birthday=None, gender='m', app_party=APP_PARTY_YANDEX,
                       related_yandex_client_id=None,
                       related_yandex_client_secret=None):
    if avatar is Undefined:
        avatar = GOOGLE_AVATAR
    links = links if links is not None else [
        'https://plus.google.com/118320684662584130204',
    ]

    app_attrs = dict(id='google-oauth2')
    if app_party is not None:
        app_attrs.update(
            dict(
                third_party=app_party is APP_PARTY_3RD_PARTY,
            ),
        )
    if related_yandex_client_id and related_yandex_client_secret:
        app_attrs.update(
            dict(
                related_yandex_client_id=related_yandex_client_id,
                related_yandex_client_secret=related_yandex_client_secret,
            ),
        )

    response = {
        'profile': {
            'username': username,
            'gender': gender,
            'links': links,
            'firstname': firstname,
            'lastname': lastname,
            'userid': userid,
            'avatar': avatar,
            'provider': {
                'code': provider_code,
                'name': provider_name,
                'id': provider_id,
            },
            'email': email,
            'birthday': birthday,
        },
        'finished': 1422607463.24,
        'task_id': task_id,
        'created': 1422607461.75,
        'yandexuid': '1046714081386436401',
        'consumer': 'passport',
        'token': {
            'scope': 'https://www.googleapis.com/auth/userinfo.profile,https://www.googleapis.com/auth/userinfo.email',
            'expires': 1422611062,
            'value': 'ya29.CwGPoL7jPi-nQNDEm3WqdSY1M-ulJLRHSz2ERTdsLLFBreAfnRzvYp441kKpKx9dhUZJskElWRokaA',
            'application': 'google-oauth2',
            'application_attributes': app_attrs,
        },
    }

    if with_business:
        response['profile']['business'] = {
            'id': FACEBOOK_BUSINESS_ID,
            'token': FACEBOOK_BUSINESS_TOKEN,
        }

    for container, key in traverse_dict_from_leaves_to_root(response):
        if container[key] is None:
            del container[key]

    return response


def task_not_found_error():
    return {
        'error': {
            'description': 'Task with the given task_id not found',
            'name': 'task-not-found',
            'request_id': '7b240321-3',
        }
    }


def social_api_v3_error_response(error):
    return dict(
        status='error',
        errors=[error],
    )


class FakeSocialApi(BaseFakeBuilder):
    def __init__(self):
        super(FakeSocialApi, self).__init__(SocialApi)

        self.set_social_api_response_value = self.set_response_value_without_method
        self.set_social_api_response_side_effect = self.set_response_side_effect_without_method

    @staticmethod
    def parse_method_from_request(http_method, url, data, headers=None):
        parsed_url = urlparse(url)
        for (http_method_name, method_pattern), method_name in iteritems(_url_path_to_method_name):
            if http_method == http_method_name and re.match(method_pattern, parsed_url.path):
                return method_name
