# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.builders.social_api.exceptions import (
    BaseSocialApiError,
    ProfileNotFoundError,
    SocialApiRequestError,
    SocialApiTemporaryError,
    SubscriptionAlreadyExistsError,
    SubscriptionNotFoundError,
    TaskNotFoundError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.social_api')


class SocialApi(BaseBuilder, JsonParserMixin):
    base_error_class = BaseSocialApiError
    temporary_error_class = SocialApiTemporaryError
    parser_error_class = SocialApiRequestError
    accept_empty_response = True

    def __init__(self, url=None, useragent=None, timeout=None, retries=None,
                 graphite_logger=None, tvm_dst_alias='social_api'):
        timeout = timeout or settings.SOCIAL_API_TIMEOUT
        retries = retries or settings.SOCIAL_API_RETRIES
        graphite_logger = graphite_logger or GraphiteLogger(service='social_api')
        super(SocialApi, self).__init__(
            url=url or settings.SOCIAL_API_URL,
            timeout=timeout,
            retries=retries,
            logger=log,
            graphite_logger=graphite_logger,
            tvm_dst_alias=tvm_dst_alias,
            useragent=useragent,
        )

    def _request_social_api(self, method, url_suffix, params=None, data=None,
                            error_detector=None):
        params = params or dict()
        params.setdefault('consumer', settings.SOCIAL_API_CONSUMER)

        error_detector = error_detector or self.check_response_for_errors

        return self._request_with_retries_simple(
            method=method,
            url_suffix=url_suffix,
            params=params,
            data=data,
            error_detector=error_detector,
            parser=self.parse_json,
        )

    def check_response_for_errors(self, data, raw_response):
        if raw_response.status_code == 409:
            raise SubscriptionAlreadyExistsError()

        if data is None or 'error' not in data:
            return

        error = data['error']
        description = error.get('description')

        name = error.get('name')
        if name == 'profile-not-found':
            raise ProfileNotFoundError()
        if name == 'subscription-not-found':
            raise SubscriptionNotFoundError()
        elif name == 'internal-exception':
            raise self.temporary_error_class(description)
        elif name == 'task-not-found':
            raise TaskNotFoundError()

        raise self.base_error_class(description)

    def check_response_v3_for_errors(self, response_dict, raw_response):
        if response_dict.get('status') != 'error':
            return

        error = response_dict['errors'][0]

        if error == 'database.failed':
            raise self.temporary_error_class(error)
        elif error == 'exception.unhandled':
            raise self.temporary_error_class(error)

        raise self.base_error_class(error)

    def get_task_data(self, task_id):
        if not task_id:
            raise TaskNotFoundError()
        return self._request_social_api(
            method='GET',
            url_suffix='task/' + task_id,
        )

    def delete_task(self, task_id):
        return self._request_social_api(
            method='DELETE',
            url_suffix='task/' + task_id,
        )

    def filter_allow_auth(self, profiles):
        filtered = [x for x in profiles if x['allow_auth'] and x['uid'] is not None]
        log.debug('Profiles with allow_auth=1 count: %d', len(filtered))
        return filtered

    def get_profiles(self, userid=None, provider=None, provider_id=None, uid=None, uids=None, allow_auth=False,
                     subscriptions=False, tokens=False, person=False, business_info=None):
        include = ','.join(
            value for (flag, value) in zip(
                [subscriptions, tokens, person],
                ['subscriptions', 'tokens', 'person'],
            ) if flag
        )
        log.info('Listing profiles for userid=%s, provider=%s, provider_id=%s, uid=%s, uids=%s, include=%s',
                 userid, provider, provider_id, uid, uids, include)

        if userid is not None and provider is None and provider_id is None:
            raise ValueError('Either provider or provider_id should be passed')
        request_args = {}
        if userid is not None:
            request_args['userid'] = userid
        if provider is not None:
            request_args['provider'] = provider
        if provider_id is not None:
            request_args['provider_id'] = provider_id
        if uid:
            request_args['uid'] = uid
        elif uids:
            request_args['uids'] = ','.join(map(str, uids))

        if business_info:
            request_args['business_token'] = business_info.token
            request_args['business_id'] = business_info.id

        if include:
            request_args['include'] = include

        response = self._request_social_api(
            method='GET',
            url_suffix='profiles',
            params=request_args,
        )

        profiles = response['profiles']
        log.debug('Social profiles count: %d', len(profiles))

        if allow_auth:
            profiles = self.filter_allow_auth(profiles)
        return profiles

    def get_profiles_by_uid(self, uid, subscriptions=False, tokens=False,
                            person=False, allow_auth=False, expand_provider=False):
        """Расширенный список социальных профилей для данного uid"""
        include = ','.join(
            value for (flag, value) in zip(
                [subscriptions, tokens, person],
                ['subscriptions', 'tokens', 'person'],
            ) if flag
        )
        log.info('Listing profiles for uid=%s with include=%s', uid, include)

        params = dict(include=include) if include else {}
        if expand_provider:
            params.update(expand='provider')
        response = self._request_social_api(
            method='GET',
            url_suffix='user/%s/profile' % uid,
            params=params,
        )
        profiles = response.get('profiles', [])

        if allow_auth:
            profiles = self.filter_allow_auth(profiles)
        return profiles

    def bind_task_profile(self, task_id, uid):
        log.info('Creating or updating social profile...')
        result = self._request_social_api(
            method='POST',
            url_suffix='task/%s/bind' % task_id,
            params={'uid': uid, 'allow_auth': 1},
        )
        return result['profile_id']

    def delete_profile(self, profile_id):
        """Удаление социального профиля"""
        log.info('Deleting profile profile_id=%s', profile_id)
        return self._request_social_api(
            method='DELETE',
            url_suffix='profile/%s' % profile_id,
        )

    def set_authentificate_profile(self, profile_id, set_auth):
        """Контроль состояния авторизации по данному профилю"""
        log.info('Set authenticate profile profile_id=%s', profile_id)
        return self._request_social_api(
            method='POST',
            url_suffix='profile/%s' % profile_id,
            data={'allow_auth': set_auth},
        )

    def delete_all_profiles_by_uid(self, uid):
        log.info('Deleting all profiles for uid=%s', uid)
        return self._request_social_api(
            method='DELETE',
            url_suffix='user/%s' % uid,
        )

    def delete_subscription(self, profile_id, sid):
        """Удаление подписки на яндексовый сервис"""
        log.info('Deleting subscription sid=%s profiles for profile_id=%s', sid, profile_id)
        return self._request_social_api(
            method='DELETE',
            url_suffix='profile/%s/subscription/%s' % (profile_id, sid),
        )

    def create_subscription(self, profile_id, sid):
        """Создание подписки на яндексовый сервис"""
        log.info('Creating subscription sid=%s profiles for profile_id=%s', sid, profile_id)
        return self._request_social_api(
            method='PUT',
            url_suffix='profile/%s/subscription/%s' % (profile_id, sid),
        )

    def delete_social_data(self, profile_ids):
        """Удаление полученных от социального провайдера данных"""
        profile_ids = [str(profile_id) for profile_id in profile_ids]
        log.info(
            'Deleting data that was gotten from social provider '
            'for profiles %s' % ', '.join(profile_ids),
        )
        return self._request_social_api(
            method='POST',
            url_suffix='delete_social_data',
            data={'profile_ids': ','.join(profile_ids)},
            error_detector=self.check_response_v3_for_errors,
        )

    def delete_tokens_from_account(self, uid, provider_name, revoke=False):
        return self._request_social_api(
            method='POST',
            url_suffix='/api/token/delete_from_account',
            data=dict(
                provider_name=provider_name,
                revoke='1' if revoke else '0',
                uid=str(uid),
            ),
        )


def get_social_api():
    return SocialApi()  # pragma: no cover


__all__ = [
    'get_social_api',
    'SocialApi',
]
