# -*- coding: utf-8 -*-
import logging
import re

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.builders.social_broker.exceptions import (
    BaseSocialBrokerError,
    SocialBrokerApplicationUnknownError,
    SocialBrokerInvalidPkceVerifierError,
    SocialBrokerInvalidTaskIdError,
    SocialBrokerInvalidTokenError,
    SocialBrokerProfileNotAllowedError,
    SocialBrokerProviderUnknownError,
    SocialBrokerRequestError,
    SocialBrokerTaskNotFoundError,
    SocialBrokerTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from six import iteritems


log = logging.getLogger('passport.social_broker')
re_avatar_key = re.compile(r'^(\d+)x(\d+)$', re.IGNORECASE)

_CODE_TO_EXCEPTION = {
    'InternalError': SocialBrokerTemporaryError,
    'DatabaseFailedError': SocialBrokerTemporaryError,
    'CommunicationFailedError': SocialBrokerTemporaryError,
    'RateLimitExceededError': SocialBrokerTemporaryError,
    'OAuthTokenInvalidError': SocialBrokerInvalidTokenError,
    'PkceVerifierInvalidError': SocialBrokerInvalidPkceVerifierError,
    'TaskNotFoundError': SocialBrokerTaskNotFoundError,
    'TaskIdInvalidError': SocialBrokerInvalidTaskIdError,
    'ApplicationUnknownError': SocialBrokerApplicationUnknownError,
    'ProviderUnknownError': SocialBrokerProviderUnknownError,
}


class SocialBroker(BaseBuilder, JsonParserMixin):
    base_error_class = BaseSocialBrokerError
    temporary_error_class = SocialBrokerTemporaryError
    parser_error_class = SocialBrokerRequestError

    def __init__(
        self,
        url=None,
        useragent=None,
        timeout=None,
        retries=None,
        graphite_logger=None,
        tvm_dst_alias='social_broker',
    ):
        timeout = timeout or settings.SOCIAL_BROKER_TIMEOUT
        retries = retries or settings.SOCIAL_BROKER_RETRIES
        graphite_logger = graphite_logger or GraphiteLogger(service='social_broker')
        super(SocialBroker, self).__init__(
            url=url or settings.SOCIAL_BROKER_URL,
            timeout=timeout,
            retries=retries,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias=tvm_dst_alias,
        )

    def check_response_for_errors(self, data, raw_response):
        error = data.get('error')
        if not error:
            return
        code = error.get('code', 'InternalError')
        message = error.get('message')

        exception_class = _CODE_TO_EXCEPTION.get(code, BaseSocialBrokerError)
        raise exception_class(message)

    def check_response_v3_for_errors(self, response_dict, raw_response):
        if response_dict.get('status') != 'error':
            return

        error = response_dict['errors'][0]

        if error == 'internal_error':
            raise self.temporary_error_class(error)
        elif error == 'exception.unhandled':
            raise self.temporary_error_class(error)
        elif error == 'profile.not_allowed':
            raise SocialBrokerProfileNotAllowedError()

        raise self.base_error_class(error)

    def get_task_by_token(self, provider, application, consumer, provider_token, provider_token_secret, scope):
        params = dict(
            provider=provider or '',
            application=application or '',
            consumer=consumer,
        )
        data = dict(
            provider_token=provider_token,
            provider_token_secret=provider_token_secret or '',
            scope=scope or '',
        )
        return self._request_with_retries_simple(
            method='POST',
            url_suffix='task_by_token',
            params=params,
            data=data,
            error_detector=self.check_response_for_errors,
            parser=self.parse_json,
        )

    def get_task_by_profile_args(self, provider, application, consumer, userid,
                                 username=None, firstname=None, lastname=None, scope=None):
        params = dict(
            consumer=consumer,
            provider=provider,
            application=application,
        )
        data = dict(
            userid=userid,
            username=username or '',
            firstname=firstname or '',
            lastname=lastname or '',
            scope=scope or '',
        )
        return self._request_with_retries_simple(
            method='POST',
            url_suffix='task_by_profile_args',
            params=params,
            data=data,
            error_detector=self.check_response_for_errors,
            parser=self.parse_json,
        )

    def check_pkce(self, task_id, code_verifier):
        if not task_id:
            raise ValueError('task_id cannot be empty')
        data = {
            'task_id': task_id,
        }
        if code_verifier:
            data.update(
                code_verifier=code_verifier,
            )
        return self._request_with_retries_simple(
            method='POST',
            url_suffix='check_pkce',
            data=data,
            error_detector=self.check_response_for_errors,
            parser=self.parse_json,
        )

    def bind_phonish_account_by_track_v2(self, consumer, uid, track_id, user_ip):
        return self._request_with_retries_simple(
            method='POST',
            url_suffix='bind_phonish_account_by_track_v2',
            data=dict(
                uid=str(uid),
                track_id=track_id,
            ),
            params=dict(consumer=consumer),
            error_detector=self.check_response_v3_for_errors,
            headers={'Ya-Consumer-Client-Ip': user_ip},
            parser=self.parse_json,
        )


def get_max_size_avatar(avatars):
    """
    Получая словарь вида {'50x0': url1, '100x200': url2, '0x0': url3...},
    вернем ссылку на картинку с наибольшим разрешением. Точные размеры без выкачивания фоток
    мы не знаем, поэтому это только эвристика. Однако, не угадать с размером и загрузить
    маленькую аватарку - не смертельно.
    """
    max_key = None
    max_size = [-1, -1]

    for key in avatars:
        result = re_avatar_key.match(key)
        if not result:
            continue
        size = sorted([int(x) for x in result.group(1, 2)], reverse=True)
        if size > max_size:
            max_size = size
            max_key = key

    if max_key:
        return avatars[max_key]


def get_best_matching_size_avatar(avatars, target_size):
    """
    Получая словарь вида {'50x0': url1, '100x200': url2, '0x0': url3...},
    вернем ссылку на картинку с разрешением, наиболее близким к запрашиваемому.
    Запрашиваемое разрешение в формате (50, 60).
    """
    if not avatars:
        return

    pairs = []
    for key, url in iteritems(avatars):
        result = re_avatar_key.match(key)
        if not result:
            continue
        size = [int(x) for x in result.group(1, 2)]
        metric = abs(size[0] - target_size[0]) + abs(size[1] - target_size[1])
        pairs.append(dict(metric=metric, url=url))

    sorted_pairs = sorted(pairs, key=lambda pair: pair['metric'])
    if sorted_pairs:
        return sorted_pairs[0]['url']
    return list(avatars.values())[0]


def get_social_broker():
    return SocialBroker()  # pragma: no cover


__all__ = [
    'get_social_broker',
    'SocialBroker',
]
