# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.builders.staff.exceptions import (
    BaseStaffError,
    StaffAuthorizationInvalidError,
    StaffEntityNotFoundError,
    StaffPermanentError,
    StaffTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from six.moves import xrange


log = logging.getLogger('passport.staff')


GROUP_TYPE_DEPARTMENT = 'department'


def staff_http_error_handler(raw_response):
    if raw_response.status_code >= 500:
        log.warning(
            u'Request failed with response=%s code=%s',
            trim_message(raw_response.content.decode('utf-8')),
            raw_response.status_code,
        )
        raise StaffTemporaryError('Server is down')
    elif raw_response.status_code == 401:
        raise StaffAuthorizationInvalidError('Invalid OAuth token')
    elif raw_response.status_code == 404:
        raise StaffEntityNotFoundError('Entity with given filters doesn\'t exist')
    elif raw_response.status_code != 200:
        raise StaffPermanentError('Bad response status code: %s' % raw_response.status_code)


class Staff(BaseBuilder, JsonParserMixin):
    base_error_class = BaseStaffError
    temporary_error_class = StaffTemporaryError
    parser_error_class = StaffPermanentError

    """
    Дока к апи: https://staff-api.yandex-team.ru/v3/persons?_doc=1
    """

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, graphite_logger=None, **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='staff')
        super(Staff, self).__init__(
            url=url or settings.STAFF_URL,
            timeout=timeout or settings.STAFF_TIMEOUT,
            retries=retries or settings.STAFF_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            **kwargs
        )

    def _make_request(self, method='GET', url_suffix='', params=None, data=None, oauth_token=None,
                      parser=None, **kwargs):
        return self._request_with_retries_simple(
            url_suffix=url_suffix,
            method=method,
            params=params,
            data=data,
            http_error_handler=staff_http_error_handler,
            parser=parser or self.parse_json,
            error_detector=None,
            headers={'Authorization': 'OAuth %s' % oauth_token},
            **kwargs
        )

    def _make_request_with_pagination(self, method='GET', url_suffix='', params=None, data=None,
                                      oauth_token=None, max_pages=50, **kwargs):
        result = []
        for page in xrange(1, max_pages + 1):
            params.update(_page=page)
            rv = self._make_request(
                url_suffix=url_suffix,
                method=method,
                params=params,
                data=data,
                oauth_token=oauth_token,
                **kwargs
            )
            result += rv['result']
            if rv['page'] >= rv['pages']:
                return result

        log.warning('Max pages loaded, stopping')
        return result

    def parse_department_id(self, raw_response):
        parsed_response = self.parse_json(raw_response)
        return parsed_response.get('department_group', {}).get('department', {}).get('id')

    def get_team(self, oauth_token, department_id=None, department_url=None, fields=None, max_pages=50):
        # Описание языка запросов: https://github.com/alonho/pql#pql
        # Как правило, запросы по department_id быстрее, поэтому предпочтительней использовать его
        params = {
            'official.is_dismissed': 'false',
            '_fields': ','.join(fields or ['uid']),
        }
        if department_id:
            params['_query'] = 'department_group.department.id==%(id)s or department_group.ancestors.department.id==%(id)s' % {
                'id': department_id,
            }
        elif department_url:
            params['_query'] = 'department_group.url==\'%(url)s\' or department_group.ancestors.url==\'%(url)s\'' % {
                'url': department_url,
            }
        else:
            raise ValueError('Either department_id or department_url is required')

        return self._make_request_with_pagination(
            url_suffix='v3/persons',
            method='GET',
            params=params,
            oauth_token=oauth_token,
            max_pages=max_pages,
        )

    def get_department_id(self, oauth_token, department_url):
        params = {
            'department_group.url': department_url,
            '_fields': 'department_group.department.id',
            '_one': 1,
        }

        return self._make_request(
            url_suffix='v3/persons',
            method='GET',
            params=params,
            oauth_token=oauth_token,
            parser=self.parse_department_id,
        )

    def get_user_info(self, oauth_token, uid, fields=None):
        params = {
            'uid': uid,
            '_fields': ','.join(fields or ['uid', 'login', 'official.is_robot']),
            '_one': 1,
        }

        return self._make_request(
            url_suffix='v3/persons',
            method='GET',
            params=params,
            oauth_token=oauth_token,
        )

    def get_user_groups(self, oauth_token, login, group_type=GROUP_TYPE_DEPARTMENT,
                        fields=None, max_pages=50):
        """
        Возвращает активные группы пользователя определенного типа.
        Если нужны предки, их необходимо тоже указать в параметре fields.
        """
        params = {
            '_fields': ','.join(fields or ['group.url', 'group.ancestors.url']),
            'person.login': login,
            'group.type': group_type,
            'group.ancestors.type': group_type,
            'group.is_deleted': 'false',
            'group.ancestors.is_deleted': 'false',
        }

        return self._make_request_with_pagination(
            url_suffix='v3/groupmembership',
            method='GET',
            params=params,
            oauth_token=oauth_token,
            max_pages=max_pages,
        )


def get_staff():
    return Staff()  # pragma: no cover
