# -*- coding: utf-8 -*-
from collections import namedtuple
import logging
import re

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.builders.suggest.exceptions import (
    SuggestError,
    SuggestSyntaxError,
    SuggestTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.core.names.en import (
    EN_FEMALE_NAMES,
    EN_MALE_NAMES,
    EN_NAMES,
)
from passport.backend.core.names.ru import (
    RU_FEMALE_NAMES,
    RU_MALE_NAMES,
    RU_NAMES,
)


log = logging.getLogger('passport.suggest.name')

FioSuggestData = namedtuple('FioSuggestData', 'firstname lastname gender')

FIO_TIMEOUT = 0.3

FIO_RETRIES = 1


class FioSuggest(BaseBuilder, JsonParserMixin):

    base_error_class = SuggestError
    temporary_error_class = SuggestTemporaryError
    parser_error_class = SuggestSyntaxError

    def __init__(self, api_url=None, useragent=None, timeout=FIO_TIMEOUT, retries=FIO_RETRIES, graphite_logger=None):
        graphite_logger = graphite_logger or GraphiteLogger(service='suggest_name')
        super(FioSuggest, self).__init__(
            url=api_url or settings.FIO_SUGGEST_API_URL,
            timeout=timeout,
            retries=retries,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
        )

    def get(self, name, fallback=True):
        try:
            suggest = self._request_with_retries_simple(
                None,
                self._parse,
                url_suffix='passport',
                params=dict(facts='passport'),
                data=name.encode('utf-8'),
                method='POST',
            )
        except (SuggestSyntaxError, SuggestTemporaryError):
            suggest = FioSuggestData(None, None, None)

        if not fallback:
            return suggest

        if not suggest.firstname or not suggest.lastname:
            firstname, lastname = get_first_and_last_name(name)
            suggest = FioSuggestData(firstname, lastname, suggest.gender)

        if not suggest.gender:
            gender = get_gender(suggest.firstname, suggest.lastname)
            suggest = FioSuggestData(suggest.firstname, suggest.lastname, gender)

        return suggest

    def _parse(self, response):
        data = self.parse_json(response).get('passport', [])

        if not data:  # {'passport': []}
            return FioSuggestData(None, None, None)

        data = data[0]

        data = data.get('BasicFio', {})

        return FioSuggestData(
            firstname=data.get('FirstName'),
            lastname=data.get('LastName'),
            gender={'m': 'male', 'f': 'female', None: None}.get(data.get('Gender'))
        )


def get_language_by_name(name):
    """
    06.07.2012 нет единого мнения насчет определения языка для имен, содержащих символы из различных языков.
    Например, `Ира Ira` или `Иосиф Stalin`.
    Пока что решили не разбирать подобные случае и считать их как undefined поведение.
    """
    if re.match(u'[а-яА-ЯёЁ]', name):
        return 'ru'
    elif re.match(u'[a-zA-Z]', name):
        return 'en'
    return 'ru'


def get_firstnames(name):
    language = get_language_by_name(name)
    if language == 'en':
        return EN_NAMES, language
    return RU_NAMES, language


def split_name(name, firstnames=None, language=None):
    firstnames = firstnames or []

    name1, delimiter, name2 = name.partition(' ')

    name1_lower = name1.lower()
    name2_lower = name2.lower()

    # по умолчанию
    firstname = name2
    lastname = name1

    if name1_lower in firstnames:
        firstname = name1
        lastname = name2
    elif language == 'ru':
        # регэкспы для определения пола, можно использовать также для определения фамилии
        if settings.MALE_SEX_REGEXP.search(name1_lower) or settings.FEMALE_SEX_REGEXP.search(name1_lower):
            firstname = name2
            lastname = name1
        elif settings.MALE_SEX_REGEXP.search(name2_lower) or settings.FEMALE_SEX_REGEXP.search(name2_lower):
            firstname = name1
            lastname = name2
    return firstname, lastname


def get_first_and_last_name(name):
    firstnames, language = get_firstnames(name)
    return split_name(name, firstnames, language)


def get_gender(firstname, lastname, language=None):
    """
    Некоторые английские имена могут быть как у мужчин и женщин.
    По одной фамилии в английском языке невозможно определить пол.
    Поэтому подобные случаи считаются особыми и остаются as is.
    То есть сейчас определение пола для таких случаев определяется порядком if.
    """
    language = language or get_language_by_name('%s %s' % (firstname, lastname))
    gender = 'unknown'
    firstname = firstname.lower()
    lastname = lastname.lower()
    if language == 'en':
        if firstname in EN_FEMALE_NAMES:
            gender = 'female'
        elif firstname in EN_MALE_NAMES:
            gender = 'male'
    elif language == 'ru':
        if firstname in RU_FEMALE_NAMES:
            gender = 'female'
        elif firstname in RU_MALE_NAMES:
            gender = 'male'
        elif settings.MALE_SEX_REGEXP.search(lastname):
            gender = 'male'
        elif settings.FEMALE_SEX_REGEXP.search(lastname):
            gender = 'female'
    return gender


def get_fio_suggest():
    return FioSuggest()
