# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger

from .exceptions import (
    BaseTakeoutError,
    TakeoutPermanentError,
    TakeoutTemporaryError,
)


log = logging.getLogger('passport.takeout')


def error_handler(response):
    if response.status_code >= 500:
        log.warning(
            u'Request failed with with response=%s code=%s',
            trim_message(response.content.decode('utf-8')),
            response.status_code,
        )
        if response.status_code == 503:
            raise TakeoutTemporaryError('Request failed')
        else:
            raise TakeoutPermanentError('Server error')


def error_detector(response, raw_response):
    if response.get('status') != 'ok':
        error_code = response.get('error')
        log.warning(u'Request failed: %s', error_code)
        raise TakeoutPermanentError(error_code)


class Takeout(BaseBuilder, JsonParserMixin):

    base_error_class = BaseTakeoutError
    temporary_error_class = TakeoutTemporaryError
    parser_error_class = TakeoutPermanentError

    def __init__(self, url=None, consumer=None, useragent=None, timeout=None, retries=None, graphite_logger=None,
                 use_tvm=True, **kwargs):
        self.consumer = consumer or settings.TAKEOUT_CONSUMER
        graphite_logger = graphite_logger or GraphiteLogger(service='takeout')
        super(Takeout, self).__init__(
            url=url or settings.TAKEOUT_URL,
            timeout=timeout or settings.TAKEOUT_TIMEOUT,
            retries=retries or settings.TAKEOUT_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias='takeout' if use_tvm else None,
            **kwargs
        )

    def _base_request(self, url_suffix, http_method, get_args=None, post_args=None):
        get_args = get_args or {}
        get_args.update(consumer=self.consumer)
        return self._request_with_retries_simple(
            url_suffix=url_suffix,
            method=http_method,
            params=get_args,
            data=post_args,
            http_error_handler=error_handler,
            parser=self.parse_json,
            error_detector=error_detector,
        )

    def prepare_archive(self, uid, unixtime):
        return self._base_request(
            url_suffix='1/prepare_archive/',
            http_method='POST',
            post_args={
                'uid': uid,
                'unixtime': unixtime,
            }
        )


def get_takeout():
    return Takeout()  # pragma: no cover
