# -*- coding: utf-8 -*-
from collections import namedtuple
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.builders.taxi_zalogin.exceptions import (
    BaseTaxiZaloginError,
    TaxiZaloginAuthEror,
    TaxiZaloginPermanentZaloginError,
    TaxiZaloginTemporaryZaloginError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from six.moves.collections_abc import Mapping


log = logging.getLogger('passport.taxi')


EVENT_TYPE = namedtuple('EventType', ['BIND', 'UNBIND'])._make(['bind', 'unbind'])


def error_detector(response, raw_response):
    # кейс, когда валидный json не является словарём
    if not isinstance(response, Mapping):
        code = ''
        message = trim_message(repr(response))
    else:
        code = response.get('code')
        message = trim_message(response.get('message', ''))
    if code == '401' or code == '403':
        # Примеры:
        # {"code":"401","message":"missing or empty X-Ya-Service-Ticket header"}
        # {"code":"403","message":"No rule found for source from tvm2 ticket"}
        log.warning('Taxi Zalogin auth error, code={}, message={}'.format(code, message))
        raise TaxiZaloginAuthEror(message)
    elif code == '400':
        # Примеры:
        # {"code":"400","message":"Missing portal_uid in query"}
        # {"code":"400","message":"Invalid JSON body: * Line 1, Column 1\n  Syntax error: value, object or array expect
        #  ... ed.\n* Line 1, Column 2\n  Extra non-whitespace after JSON value.\n"}
        log.warning('Taxi Zalogin error, code={}, message={}'.format(code, message))
        raise TaxiZaloginPermanentZaloginError(message)
    elif raw_response.status_code != 200:
        log.warning('Taxi Zalogin unexpected error, code={}, message={}'.format(code, message))
        raise TaxiZaloginTemporaryZaloginError(message)


class TaxiZalogin(BaseBuilder, JsonParserMixin):
    base_error_class = BaseTaxiZaloginError
    temporary_error_class = TaxiZaloginTemporaryZaloginError
    parser_error_class = TaxiZaloginPermanentZaloginError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, graphite_logger=None,
                 use_tvm=True, **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='taxi_zalogin')
        super(TaxiZalogin, self).__init__(
            url=url or settings.TAXI_ZALOGIN_URL,
            timeout=timeout or settings.TAXI_ZALOGIN_TIMEOUT,
            retries=retries or settings.TAXI_ZALOGIN_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias='taxi_zalogin' if use_tvm else None,
            **kwargs
        )

    def _base_request(self, url_suffix, http_method, get_args=None, post_args=None):
        get_args = get_args or {}
        return self._request_with_retries_simple(
            url_suffix=url_suffix,
            method=http_method,
            params=get_args,
            data=post_args,
            http_error_handler=None,
            parser=self.parse_json,
            error_detector=error_detector,
        )

    def uid_notify(self, portal_uid, phonish_uid, event_type):
        return self._base_request(
            url_suffix='v1/uid-notify',
            http_method='POST',
            get_args={
                'portal_uid': portal_uid,
                'phonish_uid': phonish_uid,
                'event_type': event_type,
            },
        )


def get_taxi_zalogin():
    return TaxiZalogin()  # pragma: no cover
