# -*- coding: utf-8 -*-
import logging

from google.protobuf.message import DecodeError
from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.tensornet.exceptions import (
    BaseTensorNetError,
    TensorNetInvalidResponseError,
    TensorNetMissingFeatureError,
    TensorNetTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.utils.math import sigmoid
from passport.backend.utils.string import (
    escape_unprintable_bytes,
    smart_text,
)
from quality.tsnet.protos import ts_calc_pb2


log = logging.getLogger('passport.builders.tensornet')


FEATURE_TYPE_CATEG = 'categ'


def build_eval_protobuf(features_dict, features_description):
    request = ts_calc_pb2.TMasterRequest()
    sample = request.Sample.add()
    for feature_name, feature_type in features_description:
        if feature_name not in features_dict:
            log.error('Required feature "%s" not found', feature_name)
            raise TensorNetMissingFeatureError()
        value = features_dict[feature_name]
        if feature_type == FEATURE_TYPE_CATEG:
            sample.CategFeature.append(value)
        else:
            sample.FloatFeature.append(float(value))
    return request.SerializeToString()


def http_error_handler(response):
    if response.status_code != 200:
        log.warning(
            u'Request failed with with response=%s code=%s',
            trim_message(escape_unprintable_bytes(smart_text(response.content, errors='replace'))),
            response.status_code,
        )
        raise TensorNetTemporaryError()


class TensorNet(BaseBuilder):
    """
    Билдер для общения с API TensorNet
    """

    base_error_class = BaseTensorNetError
    temporary_error_class = TensorNetTemporaryError
    parser_error_class = TensorNetInvalidResponseError

    def __init__(self, model, url=None, useragent=None, timeout=None, retries=None,
                 graphite_logger=None):
        model_config = settings.TENSORNET_MODEL_CONFIGS[model]
        timeout = timeout or model_config['timeout']
        retries = retries or model_config['retries']
        graphite_logger = graphite_logger or GraphiteLogger(
            service='tsnet',
            model='-'.join(model),
        )
        url = url or settings.TENSORNET_API_URL
        super(TensorNet, self).__init__(
            url=url,
            timeout=timeout,
            retries=retries,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
        )
        self.features_description = model_config['features_description']
        self.model_name, self.model_version = model

    def _get_url_suffix(self, handle):
        suffix = '%s/%s/%s' % (self.model_name, self.model_version, handle)
        return suffix.lstrip('/')  # Для поддержки хождения в локальный ts_calc

    def eval(self, features_dict):
        return self._request_with_retries_simple(
            method='POST',
            url_suffix=self._get_url_suffix('eval'),
            data=build_eval_protobuf(features_dict, self.features_description),
            error_detector=None,
            parser=self._parse_predict_response,
            http_error_handler=http_error_handler,
        )

    def _parse_predict_response(self, raw_response):
        response = raw_response.content
        pb_response = ts_calc_pb2.TMasterResponse()

        try:
            pb_response.ParseFromString(response)
        except DecodeError as e:
            trimmed_response = trim_message(escape_unprintable_bytes(smart_text(response, errors='replace')))
            log.warning(u'TensorNet returned invalid protobuf: %s (%s)', trimmed_response, e)
            raise self.parser_error_class(u'Invalid protobuf in TensorNet response: "%s"' % trimmed_response)

        if not pb_response.Target:
            # В случае падения slave-процесса API начинает отдавать пустой список Target
            log.warning(u'TensorNet returned empty target')
            raise self.parser_error_class(u'TensorNet returned empty target')

        return round(sigmoid(pb_response.Target[0]), 4)


def get_tensornet(model, timeout=None):
    return TensorNet(model, timeout=timeout)  # pragma: no cover
