# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.builders.trust_api.exceptions import (
    BaseTrustError,
    TrustAuthorizationInvalidError,
    TrustBadDataError,
    TrustPermanentError,
    TrustTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.trust_bindings')


VERIFICATION_STATUS_SUCCESS = 'success'
VERIFICATION_STATUS_WAIT = 'wait_for_notification'


def bind_error_detector(response, _raw_response):
    if 'binding' not in response:
        raise TrustPermanentError('No binding data in response')
    if 'id' not in response['binding']:
        raise TrustPermanentError('No binding_id in response')


def verify_error_detector(response, _raw_response):
    if 'verification' not in response:
        raise TrustPermanentError('No verification data in response')
    if 'id' not in response['verification']:
        raise TrustPermanentError('No verification_id in response')


def check_payment_error_detector(response, _raw_response):
    if 'status' not in response:
        raise TrustPermanentError('No status in response')
    if response['status'] == VERIFICATION_STATUS_WAIT and 'redirect_3ds_url' not in response:
        raise TrustTemporaryError('No redirect_3ds_url in response')


def bindings_http_error_handler(raw_response):
    if raw_response.status_code == 200:
        return

    log.warning(
        u'Request failed with response=%s code=%s',
        trim_message(raw_response.content.decode('utf-8')),
        raw_response.status_code,
    )
    if raw_response.status_code >= 500:
        raise TrustPermanentError('Server is down')
    elif raw_response.status_code == 400:
        raise TrustBadDataError('Input payload validation failed')
    elif raw_response.status_code == 401:
        raise TrustAuthorizationInvalidError('No auth')
    elif raw_response.status_code == 403:
        raise TrustAuthorizationInvalidError('Invalid auth')
    elif raw_response.status_code == 404:
        raise TrustBadDataError('Binding or verification not found')
    elif raw_response.status_code == 429:
        return TrustPermanentError('Limit exceeded')
    elif raw_response.status_code != 200:
        raise TrustPermanentError('Bad response status code: %s' % raw_response.status_code)


class TrustBinding(BaseBuilder, JsonParserMixin):
    base_error_class = BaseTrustError
    temporary_error_class = TrustTemporaryError
    parser_error_class = TrustPermanentError

    """
    Дока к апи: https://wiki.yandex-team.ru/trust/newbindingapi/#3dsverifikacijakarty
    """

    def __init__(self, url=None, timeout=None, retries=None, graphite_logger=None, useragent=None,
                 **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='trust_3ds')
        super(TrustBinding, self).__init__(
            url=url or settings.TRUST_BINDINGS_URL,
            timeout=timeout or settings.TRUST_BINDINGS_TIMEOUT,
            retries=retries or settings.TRUST_BINDINGS_RETRIES,
            logger=log,
            graphite_logger=graphite_logger,
            useragent=useragent,
            tvm_dst_alias='trust_binding_api',
            **kwargs
        )

    def _make_request(self, oauth_token, method='GET', url_suffix='', json_data=None, parser=None,
                      http_error_handler=None, error_detector=None, **kwargs):
        return self._request_with_retries_simple(
            url_suffix=url_suffix,
            method=method,
            json_data=json_data,
            headers={
                'Accept': 'application/json',
                'X-OAuth-Token': oauth_token,
            },
            parser=parser or self.parse_json,
            http_error_handler=http_error_handler,
            error_detector=error_detector,
            **kwargs
        )

    def bind(self, oauth_token, card_data_encrypted):
        """
        {
            'card_data_encrypted': 'XNJBoHEd...egPGgqw==',      - *обязательное
            'hash_algo': 'SHA512',
            'card_holder': 'Vasily Pupkin',
            'name': 'Кредитка Citi',
            'region_id': 255,
        }
        """
        bind_data = dict(
            card_data_encrypted=card_data_encrypted,
            hash_algo='SHA512',
        )

        return self._make_request(
            url_suffix='bindings/v2.0/bindings/',
            method='POST',
            json_data=bind_data,
            http_error_handler=bindings_http_error_handler,
            error_detector=bind_error_detector,
            oauth_token=oauth_token,
        )

    def verify(self, oauth_token, binding_id, card_data_encrypted, is_first=True):
        """
        {
            "method": "standard2_3ds",      - *обязательное
            "currency": "GEL",
            "country_code": "RUS",
            "card_data_encrypted": "XNJBoHEd...egPGgqw==",
            "region_id": 225,
            "user_ip": "123.123.123.123",
            "wait_for_cvn": true
        }
        """
        verify_data = dict(
            method='auto',
            card_data_encrypted=card_data_encrypted,
            wait_for_cvn=False,
        )
        if not is_first:
            verify_data.update(method='standard2_3ds')

        return self._make_request(
            url_suffix='bindings/v2.0/bindings/%s/verify/' % binding_id,
            method='POST',
            json_data=verify_data,
            http_error_handler=bindings_http_error_handler,
            error_detector=verify_error_detector,
            oauth_token=oauth_token,
        )

    def check_payment(self, oauth_token, purchase_token):
        """
        https://wiki.yandex-team.ru/trust/mobileapi/#checkpayment
        purchase_token - это verification_id из ответов предыдущих ручек
        """
        return self._make_request(
            url_suffix='check_payment',
            method='POST',
            json_data={
                'purchase_token': purchase_token,
            },
            # Если 3DS не пройден, то ручка ответит пятисоткой, но в теле будет валидный JSON с нужными полями.
            # Поэтому тут на http-статус не смотрим.
            error_detector=check_payment_error_detector,
            oauth_token=oauth_token,
        )


def get_trust_bindings():
    return TrustBinding()  # pragma: no cover
