# -*- coding: utf-8 -*-
import json
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.builders.trust_api.exceptions import (
    BaseTrustError,
    TrustAuthorizationInvalidError,
    TrustBadDataError,
    TrustPermanentError,
    TrustTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.trust_payments')


PAYMENT_STATUS_STARTED = 'started'
PAYMENT_STATUS_AUTHORIZED = 'authorized'
PAYMENT_STATUS_NOT_AUTHORIZED = 'not_authorized'
PAYMENT_STATUS_CANCELLED = 'canceled'

PAYMENT_RESP_CODE_NOT_ENOUGH_FUNDS = 'not_enough_funds'


def _common_error_detector(response, allowed_statuses=('success', )):
    if response.get('status') == 'failed':
        raise TrustTemporaryError('Trust api returned status=failed')
    elif response.get('status') not in allowed_statuses:
        raise TrustPermanentError('Trust api returned status=%s' % response.get('status'))


def get_payment_methods_error_detector(response, _raw_response):
    _common_error_detector(response)
    if 'bound_payment_methods' not in response:
        raise TrustPermanentError('No bound_payment_methods in response')


def create_basket_error_detector(response, _raw_response):
    _common_error_detector(response)
    if not response.get('purchase_token'):
        raise TrustPermanentError('No purchase_token in response')


def get_basket_status_error_detector(response, _raw_response):
    _common_error_detector(response)

    status = response.get('status')
    payment_status = response.get('payment_status')

    if not payment_status:
        raise TrustPermanentError('No payment_status in response')
    if (
        payment_status == PAYMENT_STATUS_NOT_AUTHORIZED and
        not response.get('payment_resp_code')
    ):
        raise TrustPermanentError(
            'No payment_resp_code in response. status=%s, payment_status=%s' % (status, payment_status)
        )
    if (
        payment_status == PAYMENT_STATUS_STARTED and
        not response.get('payment_url') and
        not response.get('3ds_transaction_info')
    ):
        # урл может появиться не сразу, так что ретрай поможет
        raise TrustTemporaryError(
            'Neither `3ds_transaction_info` nor `payment_url` is in response. status=%s, payment_status=%s' % (
                status, payment_status,
            )
        )


def start_payment_error_detector(response, _raw_response):
    _common_error_detector(response)


def cancel_payment_error_detector(response, _raw_response):
    _common_error_detector(response)


def payments_http_error_handler(raw_response):
    if raw_response.status_code in (200, 201):
        return

    log.warning(
        u'Request failed with response=%s code=%s',
        trim_message(raw_response.content.decode('utf-8')),
        raw_response.status_code,
    )
    if raw_response.status_code >= 500:
        raise TrustTemporaryError('Server is down (code=%s)' % raw_response.status_code)
    elif raw_response.status_code == 400:
        raise TrustBadDataError('Input payload validation failed')
    elif raw_response.status_code == 401:
        raise TrustAuthorizationInvalidError('No auth')
    elif raw_response.status_code == 403:
        raise TrustAuthorizationInvalidError('Invalid auth')
    elif raw_response.status_code == 404:
        raise TrustBadDataError('Basket or refund not found')
    elif raw_response.status_code == 429:
        return TrustPermanentError('Limit exceeded')
    else:
        raise TrustPermanentError('Bad response status code: %s' % raw_response.status_code)


class TrustPayments(BaseBuilder, JsonParserMixin):
    base_error_class = BaseTrustError
    temporary_error_class = TrustTemporaryError
    parser_error_class = TrustPermanentError

    """
    Дока к апи:
      https://wiki.yandex-team.ru/trust/Payments/
      https://wiki.yandex-team.ru/trust/payments/api/baskets/
    """

    def __init__(self, url=None, timeout=None, retries=None, graphite_logger=None, useragent=None,
                 **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='trust_3ds')
        super(TrustPayments, self).__init__(
            url=url or settings.TRUST_PAYMENTS_URL,
            timeout=timeout or settings.TRUST_PAYMENTS_TIMEOUT,
            retries=retries or settings.TRUST_PAYMENTS_RETRIES,
            logger=log,
            graphite_logger=graphite_logger,
            useragent=useragent,
            tvm_dst_alias='trust_payment_api',
            **kwargs
        )

    def _make_request(self, uid, method='GET', url_suffix='', json_data=None, parser=None,
                      error_detector=None, **kwargs):
        return self._request_with_retries_simple(
            url_suffix=url_suffix,
            method=method,
            json_data=json_data,
            headers={
                'Accept': 'application/json',
                'X-Uid': uid,
            },
            parser=parser or self.parse_json,
            http_error_handler=payments_http_error_handler,
            error_detector=error_detector,
            **kwargs
        )

    def get_payment_methods(self, uid):
        rv = self._make_request(
            url_suffix='payment-methods',
            method='GET',
            error_detector=get_payment_methods_error_detector,
            uid=uid,
        )
        return rv['bound_payment_methods']

    def create_basket(self, uid, paymethod_id, product_id, return_path, amount=1, currency='RUB',
                      use_mobile_layout=False, use_new_trust_form=False, login_id=None, is_chaas=False):
        data = dict(
            product_id=product_id,
            amount=amount,
            currency=currency,
            return_path=return_path,
            pass_params={
                'terminal_route_data': {
                    'service_force_3ds': 1,
                },
            },
            template_tag='mobile/form' if use_mobile_layout else 'desktop/form',
        )

        afs_params = {}
        if login_id:
            afs_params.update(login_id=login_id)
        if is_chaas:
            afs_params.update(chaas=True)

        if afs_params:
            data.update(afs_params=afs_params)

        if use_new_trust_form:
            data.update(
                developer_payload=json.dumps({
                    'selected_card_id': paymethod_id,
                    'auto_start_payment': True,
                    'template': 'checkout',
                    'blocks_visibility': {
                        'cardSelector': False,
                    },
                }),
            )
        else:
            data.update(
                paymethod_id=paymethod_id,
            )

        rv = self._make_request(
            url_suffix='payments',
            method='POST',
            json_data=data,
            error_detector=create_basket_error_detector,
            uid=uid,
        )
        return rv['purchase_token']

    def get_basket_status(self, uid, purchase_token):
        return self._make_request(
            url_suffix='payments/%s' % purchase_token,
            method='GET',
            error_detector=get_basket_status_error_detector,
            uid=uid,
        )

    def start_payment(self, uid, purchase_token):
        return self._make_request(
            url_suffix='payments/%s/start' % purchase_token,
            method='POST',
            error_detector=start_payment_error_detector,
            uid=uid,
        )

    def cancel_payment(self, uid, purchase_token):
        return self._make_request(
            url_suffix='payments/%s/unhold' % purchase_token,
            method='POST',
            error_detector=cancel_payment_error_detector,
            uid=uid,
        )


def get_trust_payments():
    return TrustPayments()  # pragma: no cover
