# -*- coding: utf-8 -*-
from base64 import (
    urlsafe_b64decode,
    urlsafe_b64encode,
)
import hashlib
import hmac
import logging
from time import time

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.builders.tvm.constants import (
    TVM_STATUS_BAD_CLIENT,
    TVM_STATUS_BAD_REQUEST,
    TVM_STATUS_INVALID_CREDENTIALS,
    TVM_STATUS_TEMPORARY_ERROR,
)
from passport.backend.core.builders.tvm.exceptions import (
    BaseTVMError,
    TVMBadClientError,
    TVMBadRequestError,
    TVMInvalidCredentialsError,
    TVMPermanentError,
    TVMTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.utils.string import smart_unicode


log = logging.getLogger('passport.tvm')


def tvm_http_error_handler(raw_response):
    if raw_response.status_code >= 500:
        log.warning(
            u'Request failed with response=%s code=%s',
            trim_message(raw_response.content.decode('utf-8')),
            raw_response.status_code,
        )
        raise TVMPermanentError('Server is down')


def tvm_error_detector(response, raw_response):
    if raw_response.status_code == 200:
        return
    status = response.get('status')
    error_class_mapping = {
        TVM_STATUS_TEMPORARY_ERROR: TVMTemporaryError,
        TVM_STATUS_BAD_REQUEST: TVMBadRequestError,
        TVM_STATUS_BAD_CLIENT: TVMBadClientError,
        TVM_STATUS_INVALID_CREDENTIALS: TVMInvalidCredentialsError,
    }
    error_class = error_class_mapping.get(status, TVMPermanentError)
    raise error_class('%s: %s (%s)' % (status, response.get('error'), response.get('desc')))


def tvm_verify_ssh_error_detector(response, raw_response):
    if raw_response.status_code == 200:
        return
    status = response.get('status')
    if status == TVM_STATUS_INVALID_CREDENTIALS:
        return  # в данном случае это не ошибка, а нормальный отрицательный ответ
    else:
        return tvm_error_detector(response, raw_response)


def make_ts_and_sign(client_secret, string_to_sign):
    padded_secret = client_secret + '=' * (-len(client_secret) % 4)
    try:
        decoded_secret = urlsafe_b64decode(padded_secret)
    except TypeError:
        raise ValueError('Invalid client secret')
    ts = int(time())
    sign = urlsafe_b64encode(
        hmac.new(
            key=decoded_secret,
            msg='%s%s' % (ts, string_to_sign),
            digestmod=hashlib.sha256,
        ).digest(),
    ).strip('=')
    return ts, sign


class TVM(BaseBuilder, JsonParserMixin):

    base_error_class = BaseTVMError
    temporary_error_class = TVMTemporaryError
    parser_error_class = TVMPermanentError

    def __init__(self, tvm_url=None, useragent=None, timeout=None, retries=None, graphite_logger=None, tvm_dst_alias='tvm_api', **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='tvm')
        super(TVM, self).__init__(
            url=tvm_url or settings.TVM_URL,
            timeout=timeout or settings.TVM_TIMEOUT,
            retries=retries or settings.TVM_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias=tvm_dst_alias,
            **kwargs
        )

    def _parse_verify_ssh_response(self, raw_response):
        return self.parse_json(raw_response)

    def verify_ssh(self, string_to_sign, signed_string, uid=None, login=None, public_cert=None):
        data = dict(
            ssh_sign=signed_string,
            to_sign=string_to_sign,
        )

        if uid is not None:
            data['uid'] = uid
        elif login is not None:
            data['login'] = smart_unicode(login)
        else:
            raise ValueError('uid or login is required')

        if public_cert is not None:
            data['public_cert'] = public_cert

        return self._request_with_retries_simple(
            url_suffix='2/verify_ssh',
            method='POST',
            data=data,
            http_error_handler=tvm_http_error_handler,
            parser=self._parse_verify_ssh_response,
            error_detector=tvm_verify_ssh_error_detector,
        )


def get_tvm():
    return TVM()  # pragma: no cover
