# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.builders.ufo_api.exceptions import (
    BaseUfoApiError,
    UfoApiInvalidResponseError,
    UfoApiTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.builders.ufo_api')


def http_error_handler(response):
    if response.status_code != 200:
        log.warning(
            u'Request failed with with response=%s code=%s',
            trim_message(response.content.decode('utf-8')),
            response.status_code,
        )
        raise UfoApiTemporaryError()


class UfoApi(BaseBuilder, JsonParserMixin):
    """
    Билдер для общения с сервисом Ufo
    """

    base_error_class = BaseUfoApiError
    temporary_error_class = UfoApiTemporaryError
    parser_error_class = UfoApiInvalidResponseError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, consumer=None,
                 graphite_logger=None, use_rc=None):
        timeout = timeout or settings.UFO_API_TIMEOUT
        retries = retries or settings.UFO_API_RETRIES
        graphite_logger = graphite_logger or GraphiteLogger(service='ufo')
        super(UfoApi, self).__init__(
            url=url or settings.UFO_API_URL,
            timeout=timeout,
            retries=retries,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
        )
        self.use_rc = use_rc if use_rc is not None else settings.UFO_API_USE_RC

    def profile(self, uid):
        params = {
            'uid': uid,
        }
        if self.use_rc:
            params['use_rc'] = 'yes'
        return self._request_with_retries_simple(
            method='GET',
            url_suffix='1/profile/',
            error_detector=None,
            parser=self.parse_json,
            http_error_handler=http_error_handler,
            params=params,
        )

    def phones_stats(self, phone):
        params = {
            'phone': phone.digital,
        }
        return self._request_with_retries_simple(
            method='GET',
            url_suffix='1/phones/stats/',
            error_detector=None,
            parser=self.parse_json,
            http_error_handler=http_error_handler,
            params=params,
        )


def get_ufo_api(timeout=None):
    return UfoApi(timeout=timeout)  # pragma: no cover


__all__ = [
    'get_ufo_api',
    'UfoApi',
]
