# -*- coding: utf-8 -*-
import json
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.xml_parser.xml_parser import XmlParserMixin
from passport.backend.core.builders.video.exceptions import (
    BaseVideoApiError,
    RequestTooLongError,
    VideoApiNotFoundError,
    VideoApiPermanentError,
    VideoApiTemporaryError,
    VideoApiXmlParseError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.core.xml.xml import (
    XPath,
    xpath_first,
    xpath_to_bool,
    xpath_to_int,
)
from passport.backend.utils.common import remove_none_values
from six import string_types


log = logging.getLogger('passport.video_api')

ERROR = XPath("//response/error")
QUERY = XPath("//request/query/text()")

RESULTS_GROUPING = XPath("//response/results/grouping")

URL = XPath(".//doc/url/text()")
TITLE = XPath(".//doc/title/text()")
HD = XPath(".//doc/properties/vhdbin/text()")
VIDEO_ID = XPath(".//doc/properties/videoid/text()")
THUMBNAIL = XPath(".//doc/image-properties/thumbnail-link/text()")
THUMBNAIL_WIDTH = XPath(".//doc/image-properties/thumbnail-width-original/text()")
THUMBNAIL_HEIGHT = XPath(".//doc/image-properties/thumbnail-height-original/text()")
DURATION = XPath(".//doc/properties/MediaDuration/text()")
DESCRIPTION = XPath(".//doc/passages/passage/text()")

ERROR_CODE_EMPTY_RESPONSE = '15'
ERROR_CODE_NOT_FOUND = '16'
ERROR_CODE_REQUEST_TOO_LONG = '10001'


def http_error_handler(response):
    if response.status_code != 200:
        log.warning(
            u'Request failed with with response=%s code=%s',
            trim_message(response.content.decode('utf8')),
            response.status_code,
        )
        raise VideoApiPermanentError()


class VideoSearchApi(BaseBuilder, XmlParserMixin):
    """
    Дока: https://doc.yandex-team.ru/Search/video-api/concepts/about.html
    Ошибки: https://a.yandex-team.ru/arc/trunk/arcadia/web/report/lib/YxWeb/Search/Errors.pm
    """
    base_error_class = BaseVideoApiError
    temporary_error_class = VideoApiTemporaryError
    parser_error_class = VideoApiXmlParseError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None,
                 graphite_logger=None, **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='video_api')
        super(VideoSearchApi, self).__init__(
            url=url or settings.VIDEO_API_URL,
            timeout=timeout or settings.VIDEO_API_TIMEOUT,
            retries=retries or settings.VIDEO_API_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            **kwargs
        )

    def _video_info_error_detector(self, response, raw_response):
        error = xpath_first(response, ERROR)
        query = xpath_first(response, QUERY)

        if error is not None:
            code = error.attrib.get('code', '0')
            if code in [ERROR_CODE_EMPTY_RESPONSE, ERROR_CODE_NOT_FOUND]:
                raise VideoApiNotFoundError(
                    'Video not found for query %s with %s: %s' % (
                        query, code, error.text,
                    ),
                )
            elif code == ERROR_CODE_REQUEST_TOO_LONG:
                raise RequestTooLongError(error.text)

            raise VideoApiPermanentError(
                'Video API request failed for query %s with %s: %s' % (
                    query, code, error.text,
                ),
            )

    def _video_info_response_processor(self, response):
        results = []
        grouping = xpath_first(response, RESULTS_GROUPING)
        query = xpath_first(response, QUERY)

        if grouping is None:
            return results

        for group in grouping.iterfind('.//group'):
            group_data = {
                'id': xpath_first(group, VIDEO_ID),
                'url': xpath_first(group, URL),
                'title': xpath_first(group, TITLE),
                'description': xpath_first(group, DESCRIPTION),
                'duration': xpath_to_int(group, DURATION),
                'hd': xpath_to_bool(group, HD),
                'thumbnail': xpath_first(group, THUMBNAIL),
                'thumbnail_width': xpath_to_int(group, THUMBNAIL_WIDTH),
                'thumbnail_height': xpath_to_int(group, THUMBNAIL_HEIGHT),
            }
            clean_group = remove_none_values(group_data)
            if clean_group.get('id') and clean_group.get('url'):
                results.append(clean_group)
            else:
                log.warning(
                    u'Incomplete response=%s from VideoAPI for query=%s',
                    json.dumps(clean_group),
                    trim_message(query or ''),
                )
        return results

    def video_info(self, urls):
        if urls and isinstance(urls, string_types):
            urls = [urls]
        urls = '|'.join(map(lambda x: 'url:\"%s\"' % x, urls or []))
        try:
            return self._request_with_retries_simple(
                error_detector=self._video_info_error_detector,
                parser=self.parse_xml,
                http_error_handler=http_error_handler,
                method='GET',
                url_suffix='/search/xml',
                params={
                    'type': 'video',
                    'text': urls,
                    'family': 'none',
                },
                response_processor=self._video_info_response_processor,
            )
        except VideoApiNotFoundError:
            return []


def get_video_search_api():
    return VideoSearchApi()  # pragma: no cover
