# -*- coding: utf-8 -*-
from __future__ import absolute_import

from passport.backend.core.builders.base.base import (
    BaseBuilder,
    RequestInfo,
)
from passport.backend.core.logging_utils.helpers import (
    mask_sensitive_fields,
    trim_message,
)
from passport.backend.utils.string import escape_unprintable_bytes
from six.moves import xmlrpc_client as xmlrpclib
from six.moves.urllib.parse import urljoin


class XMLRPCBaseBuilder(BaseBuilder):
    """
    Интеграция XMLRPC в паспортный билдер.
    """
    parser_error_class = None
    fault_error_class = None

    def http_error_handler(self, raw_response):
        if raw_response.status_code != 200:
            self.logger.warning(
                u'Request failed with response=%s code=%s',
                trim_message(escape_unprintable_bytes(raw_response.content)),
                raw_response.status_code,
            )
            raise self.base_error_class()

    def parse_xmlrpc(self, raw_response):
        try:
            parser, unmarshaller = xmlrpclib.getparser()
            parser.feed(raw_response.content)
            parser.close()
            return unmarshaller.close()
        except xmlrpclib.Fault as e:
            raise self.fault_error_class(e.faultCode, e.faultString)
        except Exception as e:
            self.logger.error(
                u'XMLRPC response parse failed for response=%s',
                trim_message(escape_unprintable_bytes(raw_response.content)),
                exc_info=e,
            )
            raise self.parser_error_class()

    def get_xmlrpc_handle(self, error_detector=None, response_processor=None):
        transport = BuilderBasedTransport(
            builder=self,
            error_detector=error_detector,
            response_processor=response_processor,
        )
        return ServerProxy(self.url, transport=transport)

    def get_params_description_for_method_call(self, method_name, method_params):
        """
        Возвращает байтовую строку для записи в лог (при необходимости строка будет обрезана,
        непечатаемые символы заэскейплены).
        """
        raise NotImplementedError()  # pragma: no cover


class XMLRPCRequestInfo(RequestInfo):
    def __init__(self, method_name=None, method_params_description=None, *args, **kwargs):
        super(XMLRPCRequestInfo, self).__init__(*args, **kwargs)
        self.method_name = method_name
        self.method_params_description = method_params_description

    def __str__(self):
        result = 'url: ' + self._make_url(mask_sensitive_fields(self.get_args))
        result = result + ' Method: ' + self.method_name
        if self.method_params_description:
            result = result + ' Params: ' + trim_message(escape_unprintable_bytes(self.method_params_description))
        return result


class BuilderBasedTransport(object):
    def __init__(self, builder, error_detector, response_processor):
        self.builder = builder
        self.error_detector = error_detector
        self.response_processor = response_processor

    def request(self, host, handler, data, verbose, method_name, method_params):
        """
        Метод реализует интерфейс класса xmlrpclib.Transport, напрямую из паспортного кода не вызывается.
        """
        headers = {
            'User-Agent': 'passport-xmlrpclib-wrapper',
            'Content-Type': 'text/xml',
        }
        description = self.builder.get_params_description_for_method_call(method_name, method_params)
        return self.builder._request_with_retries(
            method='POST',
            request_info=XMLRPCRequestInfo(
                url=urljoin(self.builder.url, handler),
                get_args=None,
                post_args=data,
                method_name=method_name,
                method_params_description=description,
            ),
            parser=self.builder.parse_xmlrpc,
            error_detector=self.error_detector,
            http_error_handler=self.builder.http_error_handler,
            headers=headers,
            response_processor=self.response_processor,
        )


class ServerProxy(xmlrpclib.ServerProxy):
    """
    Обертка поддерживает передачу в транспорт исходных параметров вызова,
    что необходимо для аккуратного логгирования.
    Важно: имя класса должно быть ServerProxy, т.к. мы переопределяем приватный метод и обращаемся
    к приватным атрибутам в нем.
    """
    def __request(self, methodname, params):
        # call a method on the remote server

        request = xmlrpclib.dumps(params, methodname, encoding=self.__encoding, allow_none=self.__allow_none)

        response = self.__transport.request(
            self.__host,
            self.__handler,
            request,
            verbose=self.__verbose,
            method_name=methodname,
            method_params=params,
        )

        if len(response) == 1:
            response = response[0]

        return response
