# -*- coding: utf-8 -*-

import json
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.builders.xunistater.exceptions import (
    XunistaterBaseError,
    XunistaterPermanentError,
    XunistaterTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger


logger = logging.getLogger('passport.xunistater')


class Xunistater(BaseBuilder, JsonParserMixin):
    base_error_class = XunistaterBaseError
    parser_error_class = XunistaterPermanentError
    permanent_error_class = XunistaterPermanentError
    temporary_error_class = XunistaterTemporaryError

    def __init__(self, url=None, timeout=None, retries=None, useragent=None, graphite_logger=None,
                 *args, **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='xunistater')
        super(Xunistater, self).__init__(
            url=url or settings.XUNISTATER_URL,
            timeout=timeout or settings.XUNISTATER_TIMEOUT,
            retries=retries or settings.XUNISTATER_RETRIES,
            useragent=useragent,
            logger=logger,
            graphite_logger=graphite_logger,
            *args,
            **kwargs
        )

    def _error_handler(self, raw_response):
        if raw_response.status_code >= 500:
            logger.warning(
                u'Request failed with response=%s code=%s',
                trim_message(raw_response.content.decode('utf-8')),
                raw_response.status_code,
            )
            raise self.permanent_error_class(
                u'Xunistater is down. response=%s code=%s' % (
                    trim_message(raw_response.content.decode('utf-8')),
                    raw_response.status_code,
                ),
            )
        elif raw_response.status_code < 200 or raw_response.status_code > 299:
            raise self.permanent_error_class(
                u'Request failed with response=%s code=%s' % (
                    trim_message(raw_response.content.decode('utf-8')),
                    raw_response.status_code,
                )
            )

    def _make_request(self, method='GET', url_suffix='', params=None, data=None,
                      parser=None, **kwargs):
        return self._request_with_retries_simple(
            url_suffix=url_suffix,
            method=method,
            params=params,
            data=data,
            http_error_handler=self._error_handler,
            parser=parser or self.parse_json,
            error_detector=None,
            **kwargs
        )

    def push_metrics(self, data):
        return self._make_request(
            method='POST',
            data=json.dumps(data, sort_keys=True),
        )


def get_xunistater():  # pragma: no cover
    return Xunistater()
