# -*- coding: utf-8 -*-

from datetime import datetime
import json

from passport.backend.core.builders.base.faker.fake_builder import BaseFakeBuilder
from passport.backend.core.builders.yasms import YaSms
from passport.backend.core.test.serializers import serializers
from passport.backend.utils.common import map_dict
from six.moves.urllib.parse import urlparse


DEFAULT_PHONE_NUMBER = '+79031234567'
URL_PATH_TO_METHOD_NAME = {
    u'/sendsms': u'send_sms',
}


class FakeYaSms(BaseFakeBuilder):
    def __init__(self):
        super(FakeYaSms, self).__init__(YaSms)

    @staticmethod
    def parse_method_from_request(http_method, url, data, headers=None):
        url_path = u'/' + urlparse(url).path[1:]
        return URL_PATH_TO_METHOD_NAME[url_path]


def yasms_register_response(uid=1, phone_number=u'+71231234567', phone_id=1,
                            revalidated=False, encoding=u'windows-1251'):
    if revalidated:
        action = u'revalidated'
    else:
        action = u'added'
    response = u'''
        <?xml version="1.0" encoding="%(encoding)s"?>
        <doc>
            <phone id="%(phone_id)d" number="%(phone_number)s" uid="%(uid)d"
                   %(action)s="1" />
        </doc>
    ''' % dict(
        uid=uid,
        phone_number=phone_number,
        phone_id=phone_id,
        action=action,
        encoding=encoding,
    )
    return response.strip().encode(encoding)


def yasms_send_sms_response(sms_id=1, used_gate_ids=None):
    used_gate_ids = ','.join(map(str, used_gate_ids)) if used_gate_ids else ''
    response = """<?xml version="1.0" encoding="windows-1251"?>
                  <doc>
                    <message-sent id="{sms_id}" />
                    <gates ids="{gates}" />
                  </doc>""".strip()
    return response.format(sms_id=sms_id, gates=used_gate_ids).encode(u'utf-8')


def yasms_send_sms_response__old(sms_id=1):
    response = """<?xml version="1.0" encoding="windows-1251"?>
                  <doc>
                    <message-sent id="{sms_id}" />
                  </doc>""".strip()
    return response.format(sms_id=sms_id).encode(u'utf-8')


def yasms_check_phone_response(users, bindings_count=0, binding_limit_exceeded=False):
    default_user = {
        u'uid': u'1',
        u'active': u'1',
        u'phoneid': u'12345',
        u'phone': u'+71231234567',
        u'valid': u'valid',
        u'validation_date': u'2012-12-07 16:29:58',
    }
    items = []
    for user in users:
        user = map_dict(
            user,
            {
                u'uid': serializers.number_to_str,
                u'active': serializers.bool_to_onezero,
                u'phoneid': serializers.number_to_str,
                u'validation_date': serializers.datetime_to_str,
            },
        )
        baked_user = default_user.copy()
        baked_user.update(user)

        if not baked_user[u'validation_date']:
            baked_user[u'validation_date'] = None

        items.append(baked_user)

    return json.dumps({
        u'binding_limit_exceeded': int(binding_limit_exceeded),
        u'bindings_count': bindings_count,
        u'items': items,
    }).encode(u'utf-8')


def yasms_userphones_response(uid=1, id=1, phone_number=DEFAULT_PHONE_NUMBER,
                              active=True, valid=u'msgsent',
                              autoblocked=False, permblocked=True, secure=True,
                              validation_date=None, pending_type=None, pending_phone_id=None):
    uid = serializers.number_to_str(uid)

    response = """<?xml version="1.0" encoding="windows-1251"?>
                  <doc>
                    <uid>{uid}</uid>
                    {phone}
                  </doc>""".strip()

    if validation_date is None:
        validation_date = datetime.now()

    phone_number_response = _user_phone(dict(
        id=id,
        number=phone_number,
        active=active,
        valid=valid,
        autoblocked=autoblocked,
        permblocked=permblocked,
        secure=secure,
        validation_date=validation_date,
        pending_type=pending_type,
        pending_phone_id=pending_phone_id,
    ))
    return response.format(uid=uid, phone=phone_number_response).encode(u'utf-8')


def yasms_multi_userphones_response(uid=1, phones=None, encoding=u'windows-1251'):
    uid = serializers.number_to_str(uid)

    response = """<?xml version="1.0" encoding="{encoding}"?>
                  <doc>
                    <uid>{uid}</uid>
                    {phones}
                  </doc>""".strip()

    if phones is None:
        phones = []

    phone_numbers_response = u''
    secure_count = 0
    for phone in phones:
        if phone.get(u'secure', False):
            secure_count += 1
            if secure_count > 1:
                raise ValueError(u'Only single phone number can be secure')
        phone_numbers_response += _user_phone(phone)

    return response.format(
        encoding=encoding,
        uid=uid,
        phones=phone_numbers_response,
    ).encode(encoding)


def _user_phone(params):
    default_params = {
        u'id': 1,
        u'number': DEFAULT_PHONE_NUMBER,
        u'active': True,
        u'secure': False,
        u'cyrillic': False,
        u'valid': u'msgsent',
        u'validation_date': datetime(2006, 11, 7, 15, 14, 10),
        u'validations_left': 5,
        u'autoblocked': False,
        u'permblocked': False,
        u'blocked': False,
    }
    default_params.update(params)
    params = default_params

    if params.get(u'pending_type') is not None:
        params[u'pending_phoneid'] = params.pop(u'pending_phone_id')
    else:
        params.pop(u'pending_type', None)
        params.pop(u'pending_phone_id', None)

    params = map_dict(
        params,
        {
            u'id': serializers.number_to_str,
            u'active': serializers.bool_to_onezero,
            u'secure': serializers.bool_to_onezero,
            u'cyrillic': serializers.bool_to_onezero,
            u'validation_date': serializers.datetime_to_str,
            u'validations_left': serializers.number_to_str,
            u'autoblocked': serializers.bool_to_onezero,
            u'permblocked': serializers.bool_to_onezero,
            u'blocked': serializers.bool_to_onezero,
        },
    )

    attrs = ' '.join('%s="%s"' % (k, v) for k, v in params.items())
    return '<phone %(attrs)s />' % dict(attrs=attrs)


def yasms_prolong_valid_response(uid=1, status=u'OK'):
    uid = serializers.number_to_str(uid)
    status = status.upper()

    return json.dumps({'uid': uid, 'status': status})


def yasms_error_xml_response(message, code=None, encoding=u'utf-8'):
    """
    Возможные коды ошибок можно посмотреть в passport.builder.yasms.yasms.ERROR_CODE_TO_EXC.
    """
    if code is not None:
        code_element = u'<errorcode>{code}</errorcode>'.format(code=code)
    else:
        code_element = u''
    doc = u'''
        <?xml version="1.0" encoding="{encoding}"?>
        <doc>
            <error>{message_text}</error>
            {code_element}
        </doc>
        '''.format(code_element=code_element, message_text=message, encoding=encoding)
    return doc.strip().encode(encoding)


def yasms_drop_phone_response(uid=1, status=u'OK'):
    uid = serializers.number_to_str(uid)
    status = status.upper()

    return json.dumps({u'status': status, u'uid': uid}).encode(u'utf-8')


def yasms_error_json_response(code):
    return json.dumps({u'error': code}).encode(u'utf-8')


def yasms_validations_number_of_user_phones_response(phones, encoding=u'utf-8'):
    full_attr_phones = []
    for phone in phones:
        default_phone = {
            u'number': DEFAULT_PHONE_NUMBER,
            u'validations_number': u'1',
            u'valid': u'valid',
            u'confirmed_date': u'2006-11-07 15:14:10',
            u'other_accounts': u'0',
        }
        default_phone.update(phone)
        full_attr_phones.append(default_phone)

    phone_fmt = u'''<phone number="{number}"
                           validations_number="{validations_number}"
                           valid="{valid}"
                           confirmed_date="{confirmed_date}"
                           other_accounts="{other_accounts}" />'''
    phones_xml = u''.join(
        phone_fmt.format(**phone) for phone in full_attr_phones
    )

    return (
        u'''<?xml version="1.0" encoding="{encoding}"?>
        <doc>{phones_xml}</doc>
        '''.format(encoding=encoding, phones_xml=phones_xml).encode(encoding)
    )


def yasms_have_user_once_validated_phone_response(value=u'1', reason=u'ok'):
    return (
        u'''<?xml version="1.0" encoding="utf-8"?>
            <doc>
                <have_user_once_validated_phone value="{value}"
                                                reason="{reason}" />
            </doc>
        '''.format(value=value, reason=reason).encode(u'utf-8')
    )


def yasms_remove_userphones_response(status=u'OK'):
    status = status.upper()
    return json.dumps({u'status': status})


def yasms_check_user_response(uid=1, has_current_phone=True,
                              phone_number=u'+79010000001', cyrillic=True,
                              blocked=False):
    uid = serializers.number_to_str(uid)
    has_current_phone = serializers.bool_to_yesno(has_current_phone)
    cyrillic = serializers.bool_to_yesno(cyrillic)
    blocked = serializers.bool_to_yesno(blocked)

    mandatory = u'''
        <uid>%(uid)s</uid>
        <hascurrentphone>%(has_current_phone)s</hascurrentphone>
    ''' % {u'uid': uid, u'has_current_phone': has_current_phone}

    if has_current_phone == u'yes':
        optional = u'''
            <number>%(phone_number)s</number>
            <cyrillic>%(cyrillic)s</cyrillic>
            <blocked>%(blocked)s</blocked>
        ''' % {
            u'phone_number': phone_number,
            u'cyrillic': cyrillic,
            u'blocked': blocked,
        }
    else:
        optional = u''

    response = u'''
        <?xml version="1.0" encoding="utf-8"?>
        <doc>
            %(mandatory)s
            %(optional)s
        </doc>
    ''' % {u'mandatory': mandatory, u'optional': optional}
    return response.strip().encode(u'utf-8')


def yasms_delete_phone_response(uid=5223, status=u'OK'):
    uid = serializers.number_to_str(uid)
    status = status.upper()

    return json.dumps({u'uid': uid, u'status': status})


def yasms_confirm_response(phone_id=98, phone_number=u'+79010099888', uid=30,
                           is_valid=True, is_current=True, code_checks_left=None):
    phone_id = serializers.number_to_str(phone_id)
    is_valid = serializers.bool_to_onezero(is_valid)
    is_current = serializers.bool_to_onezero(is_current)
    code_checks_left = serializers.number_to_str(code_checks_left)
    phone_number = phone_number or u''

    response = u'''<?xml version="1.0" encoding="utf-8"?>
                   <doc>
                       <phone id="%(id)s"
                              number="%(number)s"
                              uid="%(uid)s"
                              valid="%(valid)s"
                              current="%(current)s"
                              left="%(left)s" />
                   </doc>
    ''' % dict(
        id=phone_id,
        number=phone_number,
        uid=uid,
        valid=is_valid,
        current=is_current,
        left=code_checks_left,
    )
    return response.encode(u'utf-8')
