# -*- coding: utf-8 -*-

import json
import logging

from passport.backend.core.builders.base.base import (
    BaseBuilder,
    RequestInfo,
)
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.builders.mixins.xml_parser.xml_parser import XmlParserMixin
from passport.backend.core.builders.yasms.exceptions import (
    YaSmsAccessDenied,
    YaSmsDeliveryError,
    YaSmsError,
    YaSmsLimitExceeded,
    YaSmsNoSender,
    YaSmsNoText,
    YaSmsPermanentBlock,
    YaSmsPhoneNumberValueError,
    YaSmsSyntaxError,
    YaSmsTemporaryError,
    YaSmsUidLimitExceeded,
    YaSmsValueError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.core.xml.xml import (
    XPath,
    xpath_first,
)
from six.moves.urllib.parse import urljoin


log = logging.getLogger('passport.yasms')

ERROR_CODE_TO_EXC = {
    'DONTKNOWYOU': YaSmsAccessDenied,  # Запрос пришел из незнакомого места
    'NORIGHTS': YaSmsAccessDenied,  # Недостаточно прав
    'BADPHONE': YaSmsPhoneNumberValueError,  # Невалидный формат телефонного номера
    'NOUID': YaSmsValueError,  # Не указан UID пользователя
    'NOPHONE': YaSmsValueError,  # Непровалидированный телефон с таким ID не найден
    'INTERROR': YaSmsTemporaryError,  # Внутренняя ошибка, попробуйте позже
    'NOROUTE': YaSmsDeliveryError,  # Отсутствует маршрут доставки сообщений на этот номер
    'PERMANENTBLOCK': YaSmsPermanentBlock,  # Номер перманентно заблокирован (занесен в черный список администраторами Паспорта)
    'PHONEBLOCKED': YaSmsPermanentBlock,  # Номер перманентно заблокирован (занесен в черный список администраторами Паспорта)
    'LIMITEXCEEDED': YaSmsLimitExceeded,  # Превышен дневной лимит отправки сообщений на телефонный номер
    'NOTEXT': YaSmsNoText,  # Не задан текст сообщения
    'BADFROMUID': YaSmsValueError,
    'UIDLIMITEXCEEDED': YaSmsUidLimitExceeded,  # Превышено число отправок смс от uid. См. параметр from_uid
    'NOSENDER': YaSmsNoSender,  # нет sender
}

XPATH_ERROR_CODE = XPath('//doc/errorcode/text()')
XPATH_ERROR = XPath('//doc/error/text()')
XPATH_MESSAGE_SENT = XPath('//doc/message-sent')
XPATH_USED_GATES = XPath('//doc/gates')


class YaSms(BaseBuilder, JsonParserMixin, XmlParserMixin):
    base_error_class = YaSmsError
    temporary_error_class = YaSmsTemporaryError
    parser_error_class = YaSmsSyntaxError

    def __init__(self, sender=None, url=None, useragent=None, timeout=None, retries=None,
                 graphite_logger=None):
        if useragent is not None and timeout is not None:
            raise ValueError('Timeout does not make sense when useragent is defined')
        self.sender = sender or settings.YASMS_SENDER
        timeout = timeout or settings.YASMS_TIMEOUT
        graphite_logger = graphite_logger or GraphiteLogger(service='yasms')
        super(YaSms, self).__init__(
            url=url or settings.YASMS_URL,
            timeout=timeout,
            retries=retries or settings.YASMS_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias='yasms',
        )

    def _request_builder(self, path, get_params, post_params):
        return RequestInfo(urljoin(self.url, path), get_params, post_params)

    def send_sms(self, phone_number, text, text_template_params=None, from_uid=None, route='validate',
                 client_ip=None, user_agent=None,
                 caller=None, identity=None, used_gate_ids=None):
        """
        http://doc.yandex-team.ru/Passport/YaSMSDevGuide/reference/sendsms.xml
        """
        request = {
            'phone': phone_number,
            'text': text,
            'utf8': '1',
            'route': route,
        }
        if from_uid:
            request['from_uid'] = from_uid
            request['no_blackbox'] = '1'

        if self.sender:
            request['sender'] = self.sender

        if caller:
            request['caller'] = caller

        if identity:
            request['identity'] = identity

        if used_gate_ids:
            request['previous_gates'] = used_gate_ids

        if text_template_params:
            request_post = {
                'text_template_params': json.dumps(text_template_params),
            }
            method = u'POST'
        else:
            request_post = None
            method = u'GET'

        headers = {}
        if client_ip:
            headers['Ya-Consumer-Client-Ip'] = client_ip
        if user_agent:
            headers['Ya-Client-User-Agent'] = user_agent

        return self._request_with_retries(
            method,
            self._request_builder('sendsms', request, request_post),
            self.parse_send_sms_response,
            headers=headers,
        )

    def parse_send_sms_response(self, response):
        tree = self.catch_yasms_errors_xml(self.parse_xml(response))
        msg = xpath_first(tree, XPATH_MESSAGE_SENT)
        attrib = dict(msg.attrib)
        gates = xpath_first(tree, XPATH_USED_GATES)

        response = {}
        if 'id' in attrib:
            response['id'] = int(attrib['id'])
        if gates is not None:
            gates_attrib = dict(gates.attrib)
            if gates_attrib.get('ids'):
                response['used_gate_ids'] = gates_attrib['ids']
        return response

    def catch_yasms_errors_xml(self, tree):
        return self._catch_yasms_errors_xml(tree, ERROR_CODE_TO_EXC)

    def _catch_yasms_errors_xml(self, tree, mapping):
        error_code = xpath_first(tree, XPATH_ERROR_CODE)
        error = xpath_first(tree, XPATH_ERROR)
        exc_message = '%s: %s' % (error_code, error)
        if error_code:
            raise mapping.get(error_code, YaSmsError)(exc_message)
        return tree


def get_yasms():
    return YaSms()  # pragma: no cover
