# coding: utf-8

import base64
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.conf import settings
from passport.backend.core.exceptions import BaseCoreError
from passport.backend.core.logging_utils.loggers import GraphiteLogger


class YsaMirrorBaseError(BaseCoreError):
    pass


class YsaMirrorTemporaryError(YsaMirrorBaseError):
    """Временная ошибка, например, мигания, следует поретраится"""


class YsaMirrorPermanentError(YsaMirrorBaseError):
    """Скорее всего что-то конкретно не так, ретраится бесполезно"""


log = logging.getLogger('passport.ysa_mirror_api')


class YsaMirrorAPI(BaseBuilder):
    """
    API ручки выявления фродовых заходов

    https://wiki.yandex-team.ru/passport/anti-hacking/ysa/#api
    """
    base_error_class = YsaMirrorBaseError
    temporary_error_class = YsaMirrorTemporaryError
    parser_error_class = YsaMirrorPermanentError

    def __init__(self, useragent=None, timeout=None, retries=None, graphite_logger=None,
                 mobileproxy_internal_host=None,
                 **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='ysa_mirror_api')
        super(YsaMirrorAPI, self).__init__(
            url='',  # полный урл будет вычисляться в рантайме; хост, куда обращаться, заранее неизвестен.
            timeout=timeout or settings.YSA_MIRROR_API_TIMEOUT,
            retries=retries or settings.YSA_MIRROR_API_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            **kwargs
        )
        self.mobileproxy_internal_host = mobileproxy_internal_host or settings.MOBILEPROXY_INTERNAL_HOST

    def check_response_for_errors(self, response, raw_response):
        # пока нечего проверять
        pass

    def _build_url_host(self, ip_address):
        if ip_address.version == 6:
            url_host = '[' + str(ip_address) + ']'
        else:
            url_host = str(ip_address)
        return url_host

    def check_client_by_requestid_v2(self, ip_address, request_id):
        return self._request_with_retries_simple(
            error_detector=self.check_response_for_errors,
            headers={'Host': self.mobileproxy_internal_host},
            http_error_handler=self.handle_check_client_by_request_id_v2_http_error,
            method='GET',
            params={'request_id': request_id},
            parser=self.parse_check_client_by_requestid_v2_response,
            url='http://' + self._build_url_host(ip_address) + '/_check_client/fingerprint',
        )

    def handle_check_client_by_request_id_v2_http_error(self, response):
        if response.status_code // 100 == 5:
            raise YsaMirrorTemporaryError(u'Request failed with code=%s' % response.status_code)

        if response.status_code not in (200, 206):
            raise YsaMirrorPermanentError(u'Request failed with %s' % self.format_response(response))

    def parse_check_client_by_requestid_v2_response(self, raw_response):
        if raw_response.content:
            return YsaMirrorResolution.from_bytes(raw_response.content)


class YsaMirrorResolution(object):
    def __init__(self):
        self._bytes = None

    @classmethod
    def from_bytes(cls, _bytes):
        self = YsaMirrorResolution()
        self._bytes = _bytes
        return self

    @classmethod
    def from_base64(cls, encoded_string):
        _bytes = base64.standard_b64decode(encoded_string)
        return cls.from_bytes(_bytes)

    def __eq__(self, other):
        if type(other) is YsaMirrorResolution:
            return self._bytes == other._bytes
        return NotImplemented

    def __ne__(self, other):
        if type(other) is YsaMirrorResolution:
            return not self.__eq__(other)
        return NotImplemented

    __hash__ = None

    def __repr__(self):
        return '%(cls)s.%(method)s(%(value)s)' % dict(
            cls=YsaMirrorResolution.__name__,
            method=YsaMirrorResolution.from_base64.__name__,
            value=self.to_base64(),
        )

    def to_bytes(self):
        return self._bytes

    def to_base64(self):
        return base64.standard_b64encode(self.to_bytes()).decode('utf8')


def get_ysa_mirror_api():
    return YsaMirrorAPI()  # pragma: no cover
