# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import (
    BaseBuilder,
    RequestInfo,
)
from passport.backend.core.conf import settings
from passport.backend.core.exceptions import BaseCoreError
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.utils.string import smart_str
import requests


log = logging.getLogger('passport.zora')


class BaseZoraError(BaseCoreError):
    pass


class ZoraTemporaryError(BaseZoraError):
    pass


class ZoraPermanentError(BaseZoraError):
    pass


class ZoraInvalidUrlError(ZoraPermanentError):
    pass


class ZoraInternalError(ZoraTemporaryError):
    pass


INVALID_CONTENT_TYPES = [
    'application/msword',
    'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
    'application/vnd.openxmlformats-officedocument.wordprocessingml.template',
    'application/vnd.ms-word.document.macroEnabled.12',
    'application/vnd.ms-word.template.macroEnabled.12',
    'application/vnd.ms-excel',
    'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
    'application/vnd.openxmlformats-officedocument.spreadsheetml.template',
    'application/vnd.ms-excel.sheet.macroEnabled.12',
    'application/vnd.ms-excel.template.macroEnabled.12',
    'application/vnd.ms-excel.addin.macroEnabled.12',
    'application/vnd.ms-excel.sheet.binary.macroEnabled.12',
    'application/vnd.ms-powerpoint',
    'application/vnd.openxmlformats-officedocument.presentationml.presentation',
    'application/vnd.openxmlformats-officedocument.presentationml.template',
    'application/vnd.openxmlformats-officedocument.presentationml.slideshow',
    'application/vnd.ms-powerpoint.addin.macroEnabled.12',
    'application/vnd.ms-powerpoint.presentation.macroEnabled.12',
    'application/vnd.ms-powerpoint.template.macroEnabled.12',
    'application/vnd.ms-powerpoint.slideshow.macroEnabled.12',
]


def make_zora_http_error_handler(url):
    def http_error_handler(response):
        if 'zora' in response.reason and not response.headers.get('X-Yandex-Http-Code'):
            log.error('Error during request to %s, status=%s, reason=%s' %
                      (url, response.status_code, response.text))
            raise ZoraInternalError('Error during request to %s, status=%s, reason=%s' %
                                    (url, response.status_code, response.text))
    return http_error_handler


class Zora(BaseBuilder):
    base_error_class = BaseZoraError
    temporary_error_class = ZoraTemporaryError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None,
                 graphite_logger=None, max_redirects=5):
        self.proxies = {
            'http': 'http://{host}:{port}'.format(host=settings.ZORA_HOST, port=settings.ZORA_PORT),
        }
        self.source = settings.ZORA_SOURCE
        self.max_redirects = max_redirects
        graphite_logger = graphite_logger or GraphiteLogger(service='zora')
        super(Zora, self).__init__(
            url='',
            timeout=timeout or settings.ZORA_TIMEOUT,
            retries=retries or settings.ZORA_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
        )

    def _get(self, url, headers=None):
        headers = headers if headers else {}

        headers.update({
            'X-Yandex-Sourcename': self.source,
            'X-Yandex-Response-Timeout': str(self.timeout),
            'X-Yandex-Fetchoptions': 'd',
            'X-Yandex-NoCalc': '1',
            'X-Yandex-Requesttype': 'userproxy',
        })

        if url.startswith('https://'):
            url = url.replace('https://', 'http://')
            headers.update({
                'X-Yandex-Use-Https': '1',
            })

        try:
            return self._request_with_retries(
                method='GET',
                request_info=RequestInfo(smart_str(url), None, None),
                parser=lambda raw_response: raw_response,
                headers=headers,
                proxies=self.proxies,
                allow_redirects=True,
                max_redirects=self.max_redirects,
                http_error_handler=make_zora_http_error_handler(url),
            )
        except requests.TooManyRedirects:
            log.error('Failed to fetch url=%s because maximum redirects exceeded' % url)
            raise ZoraInvalidUrlError(url, 'invalid response')
        except requests.RequestException as e:
            log.error('Error %s during request to %s' % (e, url))
            raise ZoraPermanentError(url, 'invalid response')

    def get(self, url):
        response = self._get(url)
        return response.content

    def get_image(self, url):
        response = self._get(url)
        if response.status_code not in [200, 201, 202]:
            raise ZoraInvalidUrlError(url, 'status code: %s' % response.status_code)
        if not response.content:
            raise ZoraInvalidUrlError(url, 'no data')
        if (response.headers.get('content-type') in INVALID_CONTENT_TYPES or
                response.headers.get('content-type', '').startswith('text/html')):
            raise ZoraInvalidUrlError(url, 'invalid content-type %s' % response.headers.get('content-type'))
        return response.content


def get_zora():
    return Zora()  # pragma: no cover


__all__ = [
    'ZoraInternalError',
    'ZoraInvalidUrlError',
    'ZoraTemporaryError',
    'Zora',
    'BaseZoraError',
    'get_zora',
]
