# -*- coding: utf-8 -*-
import time

from passport.backend.core.cache.backend.base import BaseCache


DEFAULT_TTL = 300
DEFAULT_CULL_FREQUENCY = 2
DEFAULT_MAX_ENTRIES = 300


class LocalMemoryCache(BaseCache):
    '''
    Базовый класс для cache в памяти, умеет только ttl и ограничение по количеству
    записей в кеше
    '''

    def __init__(self,
                 ttl=DEFAULT_TTL,
                 cull_frequency=DEFAULT_CULL_FREQUENCY,
                 max_entries=DEFAULT_MAX_ENTRIES):
        self._cache = {}
        self._expire_info = {}
        self.ttl = ttl
        self.cull_frequency = cull_frequency
        self.max_entries = max_entries

    def make_key(self, key):
        return key

    def validate_key(self, key):
        return True

    def get(self, key, default=None):
        key = self.make_key(key)
        self.validate_key(key)

        exp = self._expire_info.get(key)
        if exp is None:
            return default
        elif exp > time.time():
            return self._cache[key]

        del self._cache[key]
        del self._expire_info[key]
        return default

    def set(self, key, value, timeout=None):
        key = self.make_key(key)
        self.validate_key(key)

        self._set(key, value, timeout)

    def delete(self, key):
        key = self.make_key(key)
        self.validate_key(key)

        self._delete(key)

    def has_key(self, key):
        return self.get(key) is not None

    def clear(self):
        self._cache.clear()
        self._expire_info.clear()

    def _set(self, key, value, timeout=None):
        if len(self._cache) >= self.max_entries:
            self._cull()

        if timeout is None:
            timeout = self.ttl
        self._cache[key] = value
        self._expire_info[key] = time.time() + timeout

    def _delete(self, key):
        try:
            del self._cache[key]
        except KeyError:
            pass
        try:
            del self._expire_info[key]
        except KeyError:
            pass

    def _cull(self):
        if self.cull_frequency == 0:
            self.clear()
        else:
            doomed = [k for (i, k) in enumerate(self._cache) if i % self.cull_frequency == 0]
            for k in doomed:
                self._delete(k)
