# -*- coding: utf-8 -*-
import logging

from passport.backend.core.compare.equality.comparator import (
    ALLOWED_DISTANCE_THRESHOLD,
    ALLOWED_SIMILARITY_THRESHOLD,
    FACTOR_BOOL_MATCH,
    FACTOR_BOOL_NO_MATCH,
    FACTOR_FLOAT_MATCH,
    FACTOR_FLOAT_NO_MATCH,
    FACTOR_NOT_SET,
    FuzzyCompareResult,
    FuzzyControlAnswersComparator,
    FuzzyNameComparator,
    FuzzyStrictComparator,
    FuzzyStringComparator,
    NFD_ALLOWABLE_SHRINK_RATIO,
    NFKC_ALLOWABLE_SHRINK_RATIO,
)
from passport.backend.core.utils.ip_cache import IPAddress
from six import string_types
from six.moves import xrange


log = logging.getLogger('passport.compare.compare')
REASON_REVERSED_ORDER = 'reversed_order'

EMAIL_DISTANCE_THRESHOLD = 0.8
PHONE_DISTANCE_THRESHOLD = 0.8
_default_fuzzy_name_factor = FuzzyNameComparator.compound_factor_cls()
default_fuzzy_string_factor = FuzzyStringComparator.compound_factor_cls()

# Результирующий упрощенный фактор по результатам сравнения строк
STRING_FACTOR_MATCH = 2
STRING_FACTOR_INEXACT_MATCH = 1
STRING_FACTOR_NO_MATCH = 0

STRING_FACTOR_TO_MNEMONIC = {
    STRING_FACTOR_NO_MATCH: 'no_match',
    STRING_FACTOR_INEXACT_MATCH: 'inexact_match',
    STRING_FACTOR_MATCH: 'match',
    FACTOR_NOT_SET: 'not_calculated',
}

# Веса, используемые при сравнении окружений. Совпадение yandexuid более значимо.
UA_COMPONENT_WEIGHTS = {
    'os.name': 1,
    'browser.name': 1,
    'yandexuid': 3,
}

# Факторы для сравнения окружений (ОС, браузер, yandexuid)
UA_FACTOR_FULL_MATCH = sum(UA_COMPONENT_WEIGHTS.values())  # Все три компонента совпали
UA_FACTOR_NO_MATCH = 0


def default_fuzzy_names_factor():
    return dict(
        lastname=_default_fuzzy_name_factor,
        firstname=_default_fuzzy_name_factor,
    )


def serialize_string_factor(factor):
    # представление фактора в виде списка пар (имя поля, значение поля)
    fields = [factor._fields] if isinstance(factor._fields, string_types) else factor._fields
    return [(field, getattr(factor, field)) for field in fields]


def serialize_names_factor(factor):
    formatted_factor = dict(factor)
    for name in ('firstname', 'lastname'):
        formatted_factor[name] = serialize_string_factor(factor[name])
    return formatted_factor


def compress_string_factor(factor):
    return ', '.join(['%.2f' % f if isinstance(f, float) else str(f) for f in factor])


def string_factor_sort_key(factor):
    return (
        # точное равенство исходных строк лучше других вариантов
        factor.initial_equal == FACTOR_BOOL_MATCH,
        # удовлетворение требования по отбрасыванию неиспользуемых символов
        factor.symbol_shrink >= NFKC_ALLOWABLE_SHRINK_RATIO,
        # расстояние между строками
        factor.distance,
        # при равном расстоянии, лучше то, где не применялась транслитерация
        factor.xlit_used != FACTOR_BOOL_MATCH,
    )


def name_factor_sort_key(factor):
    return (
        # точное равенство исходных строк лучше других вариантов
        factor.initial_equal == FACTOR_BOOL_MATCH,
        # удовлетворение требования по отбрасыванию неиспользуемых символов
        factor.symbol_shrink >= NFKC_ALLOWABLE_SHRINK_RATIO,
        # расстояние между строками, если удовлетворяет требованиям
        factor.distance if factor.distance >= ALLOWED_DISTANCE_THRESHOLD else 0,
        # при равном расстоянии (удовлетворяющем требованиям), лучше то, где не применялась транслитерация
        factor.xlit_used != FACTOR_BOOL_MATCH if factor.distance >= ALLOWED_DISTANCE_THRESHOLD else False,
        factor.aggressive_shrink >= NFD_ALLOWABLE_SHRINK_RATIO,
        # равенство после отбрасывания неиспользуемых в имени символов
        factor.aggressive_equal,
        # никакие критерии равенства не выполнились - отсортируем по distance
        factor.distance,
    )


def find_best_string_factor_index(factors):
    """
    Для списка строковых факторов, найти номер лучшего совпадения.
    @param factors: список факторов компаратора FuzzyStringComparator
    @return индекс лучшего фактора в списке. Для пустого списка возвращает None.
    """
    sorted_factors_with_index = sorted(
        zip(factors, xrange(len(factors))),
        key=lambda pair: string_factor_sort_key(pair[0]),
        reverse=True,
    )
    return sorted_factors_with_index[0][1] if sorted_factors_with_index else None


def find_best_names_factor_index(results):
    """
    Для списка результатов сравнения имен найти номер лучшего совпадения.
    @param results: список объектов FuzzyCompareResult, полученных при нечетком сравнении ФИО.
    @return индекс лучшего результата в списке. Для пустого списка возвращает None.
    """
    factors = [result.factors for result in results]
    sorted_factors_with_index = sorted(
        zip(
            # фактор ФИО включает в себя фактор для имени и фамилии, поэтому распаковываем фактор ФИО
            # и берем каждый индекс дважды
            [factor for names_factor in factors for factor in names_factor.values()],
            [doubled_index for index in xrange(len(factors)) for doubled_index in (index, index)],
        ),
        key=lambda pair: name_factor_sort_key(pair[0]),
        reverse=True,
    )
    return sorted_factors_with_index[0][1] if sorted_factors_with_index else None


def name_result_to_simple_factor(compare_result):
    """
    Преобразовать результат сравнения двух имен (или фамилий) в простой фактор со значениями "да/да неточное/нет".
    """
    factors = compare_result.factors
    strict_match = compare_result.status and (
        factors.initial_equal == 1 or factors.symbol_shrink == 1.0 and factors.distance == 1.0
    )
    if strict_match:
        return STRING_FACTOR_MATCH
    if compare_result.status:
        # любое другое совпадение считаем неточным
        return STRING_FACTOR_INEXACT_MATCH
    return STRING_FACTOR_NO_MATCH


def names_result_to_simple_factor(compare_result):
    """
    Преобразовать результат сравнения двух пар (имя, фамилия) в простой фактор со значениями "да/да неточное/нет".
    """
    factors = compare_result.factors
    # точным совпадением считаем совпадение, при котором либо совпали исходные строки для фамилии или имени, либо
    # строки не содержали недопустимых символов и совпали после приведения к нижнему регистру и, опционально,
    # применения транслитерации
    strict_match = compare_result.status and any(
        factors[name].initial_equal == 1 or factors[name].symbol_shrink == 1.0 and factors[name].distance == 1.0
        for name in ['lastname', 'firstname']
    )
    if strict_match:
        return STRING_FACTOR_MATCH
    # любое другое совпадение считаем неточным
    if compare_result.status:
        return STRING_FACTOR_INEXACT_MATCH
    return STRING_FACTOR_NO_MATCH


def names_pair_sort_key(simple_factors_pair):
    """
    Ключ для сортировки пар простых факторов имени и фамилии (без учета факта перестановки имени и фамилии)
    """
    return sorted(simple_factors_pair)


def string_result_to_simple_factor(compare_result):
    """
    Преобразовать результат сравнения строк в простой фактор со значениями "да/да неточное/нет".
    """
    factors = compare_result.factors
    # точным совпадением считаем совпадение, при котором либо совпали исходные строки, либо
    # строки не содержали недопустимых символов и совпали после приведения к нижнему регистру и, опционально,
    # применения транслитерации
    strict_match = compare_result.status and (
        factors.initial_equal == 1 or factors.symbol_shrink == 1.0 and factors.distance == 1.0
    )
    if strict_match:
        return STRING_FACTOR_MATCH
    # любое другое совпадение считаем неточным
    if compare_result.status:
        return STRING_FACTOR_INEXACT_MATCH
    return STRING_FACTOR_NO_MATCH


def simple_string_factor_to_mnemonic(factor):
    return STRING_FACTOR_TO_MNEMONIC.get(factor)


def _compare_names_fuzzy(name_a, name_b, comparator):
    """
    Нечеткое сравнение имен.
    @param name_a: первая строка
    @param name_b: вторая строка
    @param comparator: объект-компаратор для строк
    @return объект типа FuzzyCompareResult
    """
    result = comparator.compare(name_a, name_b)
    log.debug('_compare_names_fuzzy result for "%s" & "%s": %s reasons: %s',
              name_a, name_b, result.status, ', '.join(result.reasons))
    return result


def _compare_ordered_names(orig_names, supplied_names, comp):
    """
    Сравнение двух ФИО в заданном порядке.
    """
    firstname_result = _compare_names_fuzzy(orig_names[0], supplied_names[0], comp)
    lastname_result = _compare_names_fuzzy(orig_names[1], supplied_names[1], comp)
    return firstname_result, lastname_result


def compare_names(orig_names, supplied_names, language_code=None):
    """
    Сравнение двух ФИО с учетом транслитерации.
    Порядок имени и фамилии не важен.
    @param orig_names: кортеж вида (имя, фамилия)
    @param supplied_names: кортеж вида (имя, фамилия)
    @param language_code: двухбуквенный ISO код языка
    @return кортеж из двух строковых факторов для имени и фамилии
    """
    comp = FuzzyNameComparator(language_code)
    simple_factor_pairs = []
    for compare_results in (
            _compare_ordered_names(orig_names, supplied_names, comp),
            _compare_ordered_names(list(reversed(orig_names)), supplied_names, comp),
    ):
        simple_factor_pairs.append([name_result_to_simple_factor(result) for result in compare_results])
    best_factor = sorted(simple_factor_pairs, key=names_pair_sort_key)[-1]
    return best_factor


def compare_lastname_with_names(orig_names, supplied_lastname, language_code=None):
    """
    Проверка совпадения заданной фамилии с именем или фамилией.
    @param orig_names: кортеж вида (имя, фамилия)
    @param supplied_lastname: заданная фамилия
    @param language_code: двухбуквенный ISO код языка
    @return объект типа FuzzyCompareResult, содержащий атрибуты status (признак сравнения),
    reasons (список строк - причин такого результата), factors (набор факторов)
    """
    comp = FuzzyNameComparator(language_code)

    factors = default_fuzzy_names_factor()
    primary_result = _compare_names_fuzzy(orig_names[1], supplied_lastname, comp)
    factors['lastname'] = primary_result.factors
    if primary_result.status:
        return FuzzyCompareResult(
            True,
            primary_result.reasons,
            factors,
        )
    reversed_result = _compare_names_fuzzy(orig_names[0], supplied_lastname, comp)
    factors['firstname'] = reversed_result.factors
    if reversed_result.status:
        reversed_result.reasons.append(REASON_REVERSED_ORDER)
        return FuzzyCompareResult(
            reversed_result.status,
            reversed_result.reasons,
            factors,
        )
    return FuzzyCompareResult(
        primary_result.status,
        primary_result.reasons,
        factors,
    )


def compare_strings(orig_string, supplied_string,
                    language_code=None, distance_threshold=ALLOWED_DISTANCE_THRESHOLD):
    """
    Нечеткое сравнение двух строк.
    @param orig_string: строка, с которой сравниваем (и которой больше доверяем)
    @param supplied_string: вторая строка
    @return объект типа FuzzyCompareResult, содержащий атрибуты status (признак сравнения),
    reasons (список строк - причин такого результата), factors (набор факторов)
    """
    comp = FuzzyStringComparator(language_code, distance_threshold=distance_threshold)
    return comp.compare(orig_string, supplied_string)


def compare_answers(orig_string, supplied_string, language_code=None,
                    similarity_threshold=ALLOWED_SIMILARITY_THRESHOLD,
                    distance_threshold=ALLOWED_DISTANCE_THRESHOLD):
    """
    Нечеткое сравнение двух строк в контрольных вопросах.
    @param orig_string: строка, с которой сравниваем (и которой больше доверяем)
    @param supplied_string: вторая строка
    @return объект типа FuzzyCompareResult, содержащий атрибуты status (признак сравнения),
    reasons (список строк - причин такого результата), factors (набор факторов)
    """
    comp = FuzzyControlAnswersComparator(
        language_code,
        similarity_threshold=similarity_threshold,
        distance_threshold=distance_threshold,
    )
    return comp.compare(orig_string, supplied_string)


def _compare_strict(distance_threshold):
    """
    Нечеткое сравнение с учетом заданного порога расстояния, без транслитерации.
    """
    def _wrapper(orig_string, supplied_string):
        """
        @param orig_string: строка, с которой сравниваем (и которой больше доверяем)
        @param supplied_string: вторая строка
        """
        comp = FuzzyStrictComparator(distance_threshold=distance_threshold)
        return comp.compare(orig_string, supplied_string)
    return _wrapper


compare_emails = _compare_strict(EMAIL_DISTANCE_THRESHOLD)
compare_phones = _compare_strict(PHONE_DISTANCE_THRESHOLD)


def compare_ips(first_ip, second_ip):
    """
    Сравнить два IP-адреса.
    """
    return int(str(first_ip) == str(second_ip)) if first_ip is not None and second_ip is not None else FACTOR_NOT_SET


def compare_ip_with_subnet(ip, subnet):
    """
    Выяснить, принадлежит ли IP подсети. Подсеть - объект, поддерживающий проверку принадлежности IP, например,
    netaddr.IPNetwork.
    """
    return int(IPAddress(ip) in subnet) if ip is not None and subnet is not None else FACTOR_NOT_SET


def compare_uas(original_ua, supplied_ua_lowered):
    """
    Сравнить два UA-окружения (имя браузера, ОС, yandexuid), вычислить фактор совпадения. Фактор совпадения
    принимает значения от 0 (ни одна компонента не совпала) до 5 (все три компоненты совпали).
    Компоненты окружений могут иметь значения None, такие компоненты считаются не совпадающими с другими значениями.
    @param original_ua: первое окружение
    @param supplied_ua_lowered: второе окружение, предполагается, что его составляющие приведены к нижнему регистру
    """
    if not original_ua or not supplied_ua_lowered:
        return FACTOR_NOT_SET
    ua_factor = sum(
        int(original_ua[field].lower() == supplied_ua_lowered[field]) * UA_COMPONENT_WEIGHTS[field]
        if original_ua[field] else 0
        for field in ('os.name', 'yandexuid', 'browser.name')
    )
    return ua_factor


__all__ = (
    'compare_names',
    'compare_lastname_with_names',
    'compare_strings',
    'compare_answers',
    'compare_emails',
    'compare_phones',
    'compress_string_factor',
    'serialize_string_factor',
    'serialize_names_factor',
    'find_best_string_factor_index',
    'find_best_names_factor_index',
    'default_fuzzy_string_factor',
    'default_fuzzy_names_factor',
    'names_result_to_simple_factor',
    'string_result_to_simple_factor',
    'string_factor_sort_key',
    'name_factor_sort_key',
    'STRING_FACTOR_MATCH',
    'STRING_FACTOR_INEXACT_MATCH',
    'STRING_FACTOR_NO_MATCH',
    'FACTOR_BOOL_MATCH',
    'FACTOR_BOOL_NO_MATCH',
    'FACTOR_FLOAT_MATCH',
    'FACTOR_FLOAT_NO_MATCH',
    'FACTOR_NOT_SET',
    'simple_string_factor_to_mnemonic',
    'compare_ips',
    'compare_ip_with_subnet',
    'compare_uas',
    'UA_FACTOR_FULL_MATCH',
    'UA_FACTOR_NO_MATCH',
)
