# -*- coding: utf-8 -*-
# Описание на Вики https://wiki.yandex-team.ru/passport/python/fuzzy-compare/date-compare-factor
from datetime import (
    datetime,
    timedelta,
)
import logging

from passport.backend.core.compare.compare import (
    FACTOR_FLOAT_MATCH,
    FACTOR_FLOAT_NO_MATCH,
    FACTOR_NOT_SET,
)
from passport.backend.utils.time import safe_local_datetime_to_date


log = logging.getLogger('passport.compare.dates')


def compare_birthdays(orig_birthday, supplied_birthday):
    """
    Сравнение дней рождения. Проверяется точное совпадение.
    @param orig_birthday: первый объект ДР
    @param supplied_birthday: второй объект ДР
    @return признак совпадения (булево значение)
    """
    return orig_birthday == supplied_birthday


BIRTHDAYS_FACTOR_FULL_MATCH = 2
BIRTHDAYS_FACTOR_INEXACT_MATCH = 1
BIRTHDAYS_FACTOR_NO_MATCH = 0


def compare_birthdays_inexact(orig_birthday, supplied_birthday):
    """
    Неточное сравнение дней рождения. Возможно точное совпадение, либо совпадение без учета года.
    @param orig_birthday: первый объект ДР
    @param supplied_birthday: второй объект ДР
    @return фактор сравнения ДР
    """
    if not orig_birthday or not supplied_birthday:
        return FACTOR_NOT_SET
    orig_date = orig_birthday.date
    supplied_date = supplied_birthday.date
    if orig_date == supplied_date:
        return BIRTHDAYS_FACTOR_FULL_MATCH
    try:
        orig_date = orig_date.replace(year=supplied_date.year)
    except ValueError:
        # Исключение возможно в случае оригинального ДР 29 февраля, в таком случае совпадения точно нет
        return BIRTHDAYS_FACTOR_NO_MATCH
    if orig_date == supplied_date:
        return BIRTHDAYS_FACTOR_INEXACT_MATCH
    return BIRTHDAYS_FACTOR_NO_MATCH


# Опорные точки для случая вспоминания пользователем даты - пары (регистрация дней назад, ошибка пользователя в днях)
# для построения линейной аппроксимации. Пары соответствуют значению фактора _REFERENCE_ERROR.
_RECALLED_BY_USER_REFERENCE_POINTS = (
    (0, 0),  # Зарегистрировался сегодня - помни это точно
    (5, 1),  # Пять дней назад - можно ошибиться на 1 день
    (30, 6),
    (365, 75),
    (365 * 5, 365 * 2),
    (365 * 10, 365 * 5),
)


# Опорные точки для функции вычисления глубины, соответствующей пороговому значению фактора,
# в зависимости от времени регистрации аккаунта (см. ниже)
_EVENT_DEPTH_REFERENCE_POINTS = (
    # Пороги описаны в тикете https://st.yandex-team.ru/PASSP-11753
    (0, 1),
    (1, 1),
    (3, 3),
    (4, 3),
    (6, 3),
    (7, 4),
    (14, 8),
    (21, 14),
    (30, 15),
    (60, 30),
    (90, 30),
    (120, 45),
    (150, 45),
    (180, 60),
    (360, 60),
    (1080, 90),
    (2160, 90),
)


_REFERENCE_ERROR = 0.3

LOOSE_DATE_THRESHOLD_FACTOR = FACTOR_FLOAT_MATCH - _REFERENCE_ERROR


def _build_linear_parameters(reference_points):
    linear_parameters = []
    for i, (date_oldness, allowed_delta) in enumerate(reference_points[1:]):
        prev_date_oldness, prev_allowed_delta = reference_points[i]
        k = float(prev_allowed_delta - allowed_delta) / (prev_date_oldness - date_oldness)
        b = allowed_delta - date_oldness * k
        linear_parameters.append((date_oldness, k, b))

    return linear_parameters


def _build_approximation(linear_parameters):
    def _approximation(x):
        for (date_oldness, k, b) in linear_parameters:
            if x <= date_oldness:
                return k * x + b
        return k * x + b
    return _approximation


# Функция для вычисления допустимой ошибки пользователя в дате в зависимости от глубины даты
RECALLED_BY_USER_ALLOWED_DELTA_FUNC = _build_approximation(_build_linear_parameters(_RECALLED_BY_USER_REFERENCE_POINTS))

# Функция для вычисления глубины, соответствующей пороговому значению фактора, в зависимости от времени регистрации аккаунта
EVENT_DEPTH_THRESHOLD_FUNC = _build_approximation(_build_linear_parameters(_EVENT_DEPTH_REFERENCE_POINTS))


def _date_difference_error(date_oldness, delta):
    """
    Фактор для оценки разницы дат, значение от 0.0 до 1.0.
    """
    if delta == 0:
        return 0.0
    reference_delta = RECALLED_BY_USER_ALLOWED_DELTA_FUNC(date_oldness)
    if delta <= reference_delta:
        ratio = float(delta) / reference_delta
        return ratio * _REFERENCE_ERROR
    elif delta > date_oldness:
        return 1.0
    ratio = float(delta - reference_delta) / (date_oldness - reference_delta)
    return _REFERENCE_ERROR + ratio * LOOSE_DATE_THRESHOLD_FACTOR


def _date_difference_factor(date_oldness, delta):
    """
    Фактор для сравнения двух дат по давности даты и ошибке пользователя.
    @param date_oldness: давность оригинальной даты в днях
    @param delta: абсолютное значение ошибки пользователя относительно
    оригинальной даты
    @return значение фактора от 0.0 (нет совпадения) до 1.0 (точное совпадение).
    """
    return FACTOR_FLOAT_MATCH - _date_difference_error(date_oldness, delta)


def compare_dates_loose(orig_date, supplied_date, tz=None):
    """
    Вычисление фактора совпадения дат, учитывающего давность оригинальной даты
    (например, даты регистрации пользователя).
    @param orig_date: исходная дата (объект типа datetime)
    @param supplied_date: дата, введенная пользователем (объект типа datetime)
    @param tz: часовой пояс для введенных дат (по умолчанию - локальный)
    @return значение фактора от 0.0 до 1.0
    """
    delta = int(abs(orig_date - supplied_date).total_seconds())
    delta = delta // (24 * 60 * 60)
    now = datetime.now()
    if tz:
        # Если работаем не в локальном поясе - время "сейчас" другое
        now = safe_local_datetime_to_date(now, tz)
    date_oldness = int((now - orig_date).total_seconds()) // (24 * 60 * 60)
    if date_oldness < 0 or now < supplied_date:
        log.debug(
            'Original (%s) or supplied (%s) date is in future',
            orig_date,
            supplied_date,
        )
        return FACTOR_FLOAT_NO_MATCH
    return _date_difference_factor(date_oldness, delta)


def _calculate_linear_function_by_two_points(x1, y1, x2, y2, x):
    return float(x2 * y1 - x1 * y2 - x * (y1 - y2)) / (x2 - x1)


SECONDS_PER_DAY = timedelta(days=1).total_seconds()
DEFAULT_MAX_DEPTH_THRESHOLD = timedelta(days=365).total_seconds()


def calculate_timestamp_depth_factor(
        registration_timestamp,
        now_timestamp,
        event_timestamp,
        max_depth_threshold=DEFAULT_MAX_DEPTH_THRESHOLD,
        fixed_threshold=None,
):
    """
    Вычисление фактора глубины timestamp-а с учетом порога, вычисляемого по глубине регистрации (либо явно заданного).
    @param registration_timestamp: timestamp регистрации
    @param now_timestamp: timestamp сейчас
    @param event_timestamp: целевой timestamp для вычисления фактора глубины
    @param max_depth_threshold: ограничение по глубине регистрации, после которого значение фактора не увеличивается
    @param fixed_threshold: фиксированный порог в секундах. Если не задан, используется переменный порог, зависящий
    от глубины регистрации
    @return значение фактора от 0.0 до 1.0
    """
    if not(registration_timestamp <= event_timestamp <= now_timestamp):
        # некорректные значения параметров
        return FACTOR_NOT_SET
    registration_depth = now_timestamp - registration_timestamp
    event_depth = now_timestamp - event_timestamp
    threshold_seconds = EVENT_DEPTH_THRESHOLD_FUNC(
        float(registration_depth) / SECONDS_PER_DAY,
    ) * SECONDS_PER_DAY if fixed_threshold is None else fixed_threshold

    if event_depth <= threshold_seconds:
        return max(
            _calculate_linear_function_by_two_points(
                0,
                FACTOR_FLOAT_NO_MATCH,
                threshold_seconds,
                LOOSE_DATE_THRESHOLD_FACTOR,
                event_depth,
            ),
            FACTOR_FLOAT_NO_MATCH,
        )
    if event_depth >= max_depth_threshold:
        return FACTOR_FLOAT_MATCH

    return min(
        _calculate_linear_function_by_two_points(
            threshold_seconds,
            LOOSE_DATE_THRESHOLD_FACTOR,
            min(max_depth_threshold, registration_depth),
            FACTOR_FLOAT_MATCH,
            event_depth,
        ),
        FACTOR_FLOAT_MATCH,
    )


def calculate_timestamp_interval_factor(
        registration_timestamp,
        now_timestamp,
        first_timestamp,
        last_timestamp,
        max_depth_threshold=DEFAULT_MAX_DEPTH_THRESHOLD,
):
    """
    Вычисление фактора длительности интервала во времени жизни аккаунта с учетом ограничения по глубине.
    @param registration_timestamp: timestamp регистрации
    @param now_timestamp: timestamp сейчас
    @param first_timestamp: первое значение timestamp интервала
    @param last_timestamp: последнее значение timestamp интервала
    @param max_depth_threshold ограничение по глубине рассматриваемых timestamp-ов
    """
    return float(
        min(max_depth_threshold, last_timestamp - first_timestamp),
    ) / min(max_depth_threshold, now_timestamp - registration_timestamp)
