# -*- coding: utf-8 -*-
"""
См. описание всего на Вики https://wiki.yandex-team.ru/passport/python/fuzzy-compare/sravneniefio
Описание вычисления факторов: https://wiki.yandex-team.ru/passport/python/fuzzy-compare/compare-factor
"""
from collections import namedtuple
import logging
import unicodedata

from namedlist import namedlist
from passport.backend.core.compare.equality.trigrams import trigrams_compare
from passport.backend.core.compare.equality.xlit_compare import (
    xlit_basic,
    xlit_compare,
)
from py2casefold import casefold
import pyxdameraulevenshtein
import six
from six import add_metaclass


log = logging.getLogger('passport.compare.equality.comparator')
FuzzyCompareResult = namedtuple(
    'FuzzyCompareResult',
    ['status', 'reasons', 'factors'],
)

REASON_NO_MATCH = 'no_match'
REASON_NO_MATCH_EMPTY = 'empty'
REASON_NO_MATCH_EMPTY_NORMALIZED = 'empty_normalized'
REASON_MATCH = 'equal'
REASON_MATCH_SUFFIX = 'equal_suffixes'

FACTOR_NOT_SET = -1

FACTOR_BOOL_MATCH = 1
FACTOR_BOOL_NO_MATCH = 0

FACTOR_FLOAT_MATCH = 1.0
FACTOR_FLOAT_NO_MATCH = 0.0

NFKC_ALLOWABLE_SHRINK_RATIO = 0.6
NFD_ALLOWABLE_SHRINK_RATIO = 0.4
ALLOWED_DISTANCE_THRESHOLD = 0.75

ALLOWED_SIMILARITY_THRESHOLD = 0.5
ALLOWED_TRANSLIT_SIMILARITY_THRESHOLD = 0.4


def _get_reason(comparator, reason):
    # возвращаем причину в виде списка из одной строки
    return ['%s.%s' % (comparator.__name__, reason)]


class CompoundFactorFieldsMetaclass(type):
    """
    Метакласс генерирует для каждого класса в иерархии класс-namedlist
    compound_factor_cls для работы с именами факторов напрямую.
    Имена факторов собираются из атрибутов factor_fields всех классов
    в порядке обхода иерархии.
    """
    def __init__(comparator_cls, *args):
        super(CompoundFactorFieldsMetaclass, comparator_cls).__init__(*args)

        whole_factor_fields = []
        for cls in comparator_cls.mro():
            cls_factors = cls.__dict__.get('factor_fields', [])
            # один и тот же фактор может использоваться в различных классах иерархии
            unique_cls_factors = [f for f in cls_factors if f not in whole_factor_fields]
            whole_factor_fields.extend(unique_cls_factors)
        comparator_cls.compound_factor_cls = namedlist(
            'CompoundFactor',
            whole_factor_fields,
            default=FACTOR_NOT_SET,
        )


@add_metaclass(CompoundFactorFieldsMetaclass)
class FuzzyComparatorBase(object):
    """
    Базовый класс для нечеткого сравнения на равенство.
    Потомки переопределяют метод _compare, в котором не забывают вызвать
    super(cls, self)._compare(a, b). Метод должен возвращать FuzzyCompareResult,
    в котором поле status отражает результат сравнения, поле reasons - список
    причин результата.

    Для работы с факторами используется метакласс, объединяющий атрибуты factor_fields
    в иерархии классов и генерирующий тип составного фактора.
    В конкретном классе-компараторе задается список factor_fields с именами полей составного
    фактора, задаваемыми в данном классе. Задать значение можно так:
    self.factor.<factor_name> = <factor_value>.
    """

    def reset_factor(self):
        self.factor = self.compound_factor_cls()

    def __init__(self, language_code=None):
        """
        Конструктор.
        @param language_code: двухбуквенный ISO код языка
        """
        self.language_code = language_code

    def _compare(self, a, b):
        # окончание цепочки вызовов - нет совпадения
        return FuzzyCompareResult(
            False,
            _get_reason(FuzzyComparatorBase, REASON_NO_MATCH),
            self.factor,
        )

    def compare(self, a, b):
        self.reset_factor()
        return self._compare(a, b)


class InitialComparator(FuzzyComparatorBase):
    """
    Базовое сравнение на равенство исходных строк.
    Также проверяет исходные строки на пустоту.
    """
    factor_fields = ['initial_equal']

    def _compare(self, a, b):
        self.factor.initial_equal = FACTOR_BOOL_NO_MATCH
        if not a or not b:
            return FuzzyCompareResult(
                False,
                _get_reason(InitialComparator, REASON_NO_MATCH_EMPTY),
                self.factor,
            )
        elif a == b:
            self.factor.initial_equal = FACTOR_BOOL_MATCH
            return FuzzyCompareResult(
                True,
                _get_reason(InitialComparator, REASON_MATCH),
                self.factor,
            )
        return super(InitialComparator, self)._compare(a, b)


def _is_shrink_allowed(shrink_factor, allowable_ratio):
    """
    Проверка того, допускаем ли мы заданное уменьшение длины строки.
    """
    return shrink_factor >= allowable_ratio


def _get_shrink_factor(a, a_new, b, b_new):
    """
    Получение фактора для случая уменьшения длин строк по какому-либо правилу.
    Подразумевается, что a_new и b_new имеют длину, не большую длин a, b соответственно.
    @return минимум отношения сокращенной длины к исходной длине для заданных строк
    """
    return min(
        len(a_new) / float(len(a)) if len(a) else FACTOR_FLOAT_NO_MATCH,
        len(b_new) / float(len(b)) if len(b) else FACTOR_FLOAT_NO_MATCH,
    )


class NFKCLowerConverter(FuzzyComparatorBase):
    """
    Преобразование в NFKC и удаление символов категорий C (Other),
    S (Symbols), Zl (separator, line), Zp (separator, paragraph). Остаются символы
    категорий L (letters), M (marks), N (numbers), Zs (separator, space), P (punctuation).
    См. страницу на Вики.
    """
    FILTERED_OUT_CATEGORIES = {'Zl', 'Zp', 'Cc', 'Cf', 'Cs', 'Co', 'Cn', 'Sm', 'Sc', 'Sk', 'So'}

    factor_fields = ['symbol_shrink']

    def _to_nfkc(self, s):
        """
        Нормализовать строку в форму NFKC.
        @param s: строка типа unicode
        @return преобразованная строка типа unicode
        """
        return unicodedata.normalize('NFKC', six.text_type(s))

    def _to_filtered_casefold(self, s):
        """
        Отбросить часть неиспользуемых в обычных строках символов,
        привести в вид для сравнения без учета регистра.
        @param s: строка типа unicode
        @return преобразованная строка типа unicode
        """
        filtered = u''.join(
            c
            for c in six.text_type(s)
            if unicodedata.category(c) not in self.FILTERED_OUT_CATEGORIES
        )
        return casefold(filtered)

    def _compare(self, a, b):
        a, b = self._to_nfkc(a), self._to_nfkc(b)
        a_filtered, b_filtered = self._to_filtered_casefold(a), self._to_filtered_casefold(b)
        log.debug(u'NFKCLowerConverter: NFKC lowered strings: "%s" & "%s"', a_filtered, b_filtered)

        shrink_factor = _get_shrink_factor(a, a_filtered, b, b_filtered)
        self.factor.symbol_shrink = shrink_factor

        # Вычисляем результат последующих классов в иерархии до проверки допустимости
        # сокращения длины для заполнения факторов.
        result = super(NFKCLowerConverter, self)._compare(a_filtered, b_filtered)
        if not _is_shrink_allowed(shrink_factor, NFKC_ALLOWABLE_SHRINK_RATIO):
            log.debug(
                u'NFKCLowerConverter: reduced original strings ("%s", "%s") too much',
                a,
                b,
            )
            return FuzzyCompareResult(
                False,
                _get_reason(NFKCLowerConverter, REASON_NO_MATCH_EMPTY_NORMALIZED),
                self.factor,
            )
        return result


def _is_distance_allowed(distance_factor, distance_threshold):
    """
    Проверка допустимого числа опечаток для строки заданной длины.
    """
    return distance_factor >= distance_threshold


def _get_distance_factor(length, distance):
    """
    Получение фактора для вычисленного расстояния Дамерау-Левенштейна.
    @param length: длина исходной строки
    @param distance: вычисленное расстояние ДЛ
    """
    return float(length) / (length + distance) if length + distance else FACTOR_FLOAT_NO_MATCH


def _compute_and_check_distance(a, b, matched_len=0, distance_threshold=ALLOWED_DISTANCE_THRESHOLD):
    """
    Вычисление расстояния Дамерау-Левенштейна между строками и проверка совпадения в
    соответствии с правилами для строк различной длины
    @param a: первая строка
    @param b: вторая строка
    @param matched_len: длина уже сматченного и отброшенного префикса строк
    @param distance_threshold: ограничение для допустимого фактора
    @return кортеж: признак совпадения, фактор совпадения
    """
    distance = pyxdameraulevenshtein.damerau_levenshtein_distance(a, b)
    length = len(a) + matched_len
    distance_factor = _get_distance_factor(length, distance)
    log.debug(u'Distance between "%s" & "%s" is %g (factor: %g)', a, b, distance, distance_factor)
    return _is_distance_allowed(distance_factor, distance_threshold), distance_factor


class DistanceComparator(FuzzyComparatorBase):
    """
    Сравнение строк с учетом расстояния Дамерау-Левенштейна.
    """
    factor_fields = ['distance']

    def __init__(self, language_code=None, distance_threshold=ALLOWED_DISTANCE_THRESHOLD):
        """
        Конструктор.
        @param distance_threshold: ограничение для допустимого фактора
        """
        super(DistanceComparator, self).__init__(language_code=language_code)
        self.distance_threshold = distance_threshold

    def _compare(self, a, b):
        status, distance_factor = _compute_and_check_distance(a, b, distance_threshold=self.distance_threshold)
        self.factor.distance = max(self.factor.distance, distance_factor)
        if status:
            return FuzzyCompareResult(
                True,
                _get_reason(DistanceComparator, REASON_MATCH),
                self.factor,
            )

        return super(DistanceComparator, self)._compare(a, b)


def _compute_and_check_similarity(a, b, similarity_threshold=ALLOWED_SIMILARITY_THRESHOLD):
    """
    Вычисление похожесть между двумя строками по триграммам
    @param a: первая строка
    @param b: вторая строка
    @param similarity_threshold: ограничение для допустимого фактора
    @return кортеж: признак совпадения, фактор совпадения
    """
    similarity_factor = trigrams_compare(a, b)
    log.debug(u'Similarity between "%s" & "%s" is %g (tr: %g)', a, b, similarity_factor, similarity_threshold)
    return (similarity_factor >= similarity_threshold), similarity_factor


class TrigramsAndDistanceComparator(FuzzyComparatorBase):
    """
    Сравнение строк по совпадению триграмм и с учетом расстояния Дамерау-Левенштейна.
    """
    factor_fields = ['distance', 'similarity']

    def __init__(self, language_code=None,
                 similarity_threshold=ALLOWED_SIMILARITY_THRESHOLD,
                 distance_threshold=ALLOWED_DISTANCE_THRESHOLD):
        """
        Конструктор.
        @param similarity_threshold: ограничение для допустимого фактора похожести
        @param distance_threshold: ограничение для допустимого фактора дистанции
        """
        super(TrigramsAndDistanceComparator, self).__init__(language_code=language_code)
        self.similarity_threshold = similarity_threshold
        self.distance_threshold = distance_threshold

    def _compare(self, a, b):
        sim_status, factor_similarity = _compute_and_check_similarity(a, b, similarity_threshold=self.similarity_threshold)
        dist_status, factor_distance = _compute_and_check_distance(a, b, distance_threshold=self.distance_threshold)

        self.factor.similarity = max(self.factor.similarity, factor_similarity)
        self.factor.distance = max(self.factor.distance, factor_distance)

        if sim_status and dist_status:
            return FuzzyCompareResult(
                True,
                _get_reason(TrigramsAndDistanceComparator, REASON_MATCH),
                self.factor,
            )

        return super(TrigramsAndDistanceComparator, self)._compare(a, b)


class TransliteratedAndDistanceComparator(FuzzyComparatorBase):
    """
    Сравнение на основе применения возможных транслитераций.
    В случае отсутствия совпадения на основе транслитераций, выполняется проверка
    расстояния для транслитерированных строк.
    """
    factor_fields = ['distance', 'xlit_used']
    FACTOR_XLIT_COMPUTED = 0
    FACTOR_XLIT_USED = 1

    def __init__(self, distance_threshold=ALLOWED_DISTANCE_THRESHOLD, **kwargs):
        super(TransliteratedAndDistanceComparator, self).__init__(**kwargs)
        self.distance_threshold = distance_threshold

    def _compare(self, a, b):
        status, suffix_a, suffix_b = xlit_compare(a, b)
        self.factor.xlit_used = self.FACTOR_XLIT_COMPUTED
        if status:
            # Точное сведение транслитерацией
            self.factor.xlit_used = self.FACTOR_XLIT_USED
            self.factor.distance = FACTOR_FLOAT_MATCH
            return FuzzyCompareResult(
                True,
                _get_reason(TransliteratedAndDistanceComparator, REASON_MATCH),
                self.factor,
            )
        matched_len = min(len(a) - len(suffix_a), len(b) - len(suffix_b))
        suffix_a, suffix_b = xlit_basic(suffix_a), xlit_basic(suffix_b)
        log.debug(u'TransliteratedAndDistanceComparator: suffixes "%s" & "%s"', suffix_a, suffix_b)
        status, distance_factor = _compute_and_check_distance(
            suffix_a,
            suffix_b,
            matched_len=matched_len,
            distance_threshold=self.distance_threshold,
        )
        if distance_factor > self.factor.distance:
            # Обновляем фактор, только если получили большее значение, чем уже есть
            self.factor.distance = distance_factor
            self.factor.xlit_used = self.FACTOR_XLIT_USED
        if status:
            return FuzzyCompareResult(
                True,
                _get_reason(TransliteratedAndDistanceComparator, REASON_MATCH_SUFFIX),
                self.factor,
            )
        return super(TransliteratedAndDistanceComparator, self)._compare(a, b)


class TransliteratedTrigramsAndDistanceComparator(FuzzyComparatorBase):
    """
    Сравнение на основе применения возможных транслитераций.
    В случае отсутствия совпадения на основе транслитераций, выполняется проверка
    похожести по триграммам и расстояниям для транслитерированных строк.
    """
    factor_fields = ['translit_similarity', 'distance', 'xlit_used']
    FACTOR_XLIT_COMPUTED = 0
    FACTOR_XLIT_USED = 1

    def __init__(self,
                 translit_similarity_threshold=ALLOWED_TRANSLIT_SIMILARITY_THRESHOLD,
                 distance_threshold=ALLOWED_DISTANCE_THRESHOLD,
                 **kwargs):
        super(TransliteratedTrigramsAndDistanceComparator, self).__init__(**kwargs)
        self.distance_threshold = distance_threshold
        self.translit_similarity_threshold = translit_similarity_threshold

    def _compare(self, a, b):
        status, suffix_a, suffix_b = xlit_compare(a, b)
        self.factor.xlit_used = self.FACTOR_XLIT_COMPUTED
        if status:
            # Точное сведение транслитерацией
            self.factor.xlit_used = self.FACTOR_XLIT_USED
            self.factor.distance = FACTOR_FLOAT_MATCH
            return FuzzyCompareResult(
                True,
                _get_reason(TransliteratedTrigramsAndDistanceComparator, REASON_MATCH),
                self.factor,
            )

        matched_len = min(len(a) - len(suffix_a), len(b) - len(suffix_b))
        suffix_a, suffix_b = xlit_basic(suffix_a), xlit_basic(suffix_b)

        sim_status, factor_similarity = _compute_and_check_similarity(
            xlit_basic(a),
            xlit_basic(b),
            similarity_threshold=self.translit_similarity_threshold,
        )
        if factor_similarity > self.factor.translit_similarity:
            self.factor.translit_similarity = factor_similarity

        log.debug(u'TransliteratedTrigramsAndDistanceComparator: suffixes "%s" & "%s"', suffix_a, suffix_b)
        dist_status, distance_factor = _compute_and_check_distance(
            suffix_a,
            suffix_b,
            matched_len=matched_len,
            distance_threshold=self.distance_threshold,
        )
        if distance_factor > self.factor.distance:
            # Обновляем фактор, только если получили большее значение, чем уже есть
            self.factor.distance = distance_factor
            self.factor.xlit_used = self.FACTOR_XLIT_USED
        if dist_status and sim_status:
            return FuzzyCompareResult(
                True,
                _get_reason(TransliteratedTrigramsAndDistanceComparator, REASON_MATCH_SUFFIX),
                self.factor,
            )
        return super(TransliteratedTrigramsAndDistanceComparator, self)._compare(a, b)


class NFDFilterComparator(FuzzyComparatorBase):
    """
    Преобразование в NFD и удаление всего кроме букв - в качестве последней
    попытки что-нибудь сматчить.
    Нужно применять с осторожностью, только в случае, если в строке обязательны буквы.
    См. страницу на Вики.
    """
    factor_fields = ['aggressive_shrink', 'aggressive_equal']

    def _nfd_filter(self, s):
        """
        Декомпозировать строку, удалить все кроме букв.
        @param s: строка типа unicode
        @return преобразованная строка типа unicode
        """
        s = unicodedata.normalize('NFD', six.text_type(s))
        return u''.join(c for c in s if unicodedata.category(c)[0] == 'L')

    def _compare(self, a, b):
        a_filtered = self._nfd_filter(a)
        b_filtered = self._nfd_filter(b)
        log.debug(u'NFDFilterComparator: NFD "%s" & "%s"', a_filtered, b_filtered)
        self.factor.aggressive_shrink = _get_shrink_factor(a, a_filtered, b, b_filtered)
        self.factor.aggressive_equal = int(a_filtered == b_filtered)
        if not _is_shrink_allowed(self.factor.aggressive_shrink, NFD_ALLOWABLE_SHRINK_RATIO):
            # если строки оказались сильно короче - скорее всего, этот компаратор
            # вообще нельзя применять для этой категории данных (либо данные кривые)
            log.debug(
                u'NFDFilterComparator: reduced original strings ("%s", "%s") too much',
                a,
                b,
            )
            return FuzzyCompareResult(
                False,
                _get_reason(NFDFilterComparator, REASON_NO_MATCH_EMPTY_NORMALIZED),
                self.factor,
            )

        if self.factor.aggressive_equal:
            return FuzzyCompareResult(
                True,
                _get_reason(NFDFilterComparator, REASON_MATCH),
                self.factor,
            )
        return super(NFDFilterComparator, self)._compare(a_filtered, b_filtered)


class FuzzyNameComparator(
    InitialComparator,
    NFKCLowerConverter,
    DistanceComparator,
    TransliteratedAndDistanceComparator,
    NFDFilterComparator,
):
    """
    Класс для нечеткого сравнения имен, основанный на применении нескольких
    преобразований и способов сравнения.
    """


class FuzzyStringComparator(
    InitialComparator,
    NFKCLowerConverter,
    DistanceComparator,
    TransliteratedAndDistanceComparator,
):
    """
    Класс для нечеткого сравнения произвольных строк, основанный на применении нескольких
    преобразований и способов сравнения. Не используется NFD.
    """


class FuzzyControlAnswersComparator(
    InitialComparator,
    NFKCLowerConverter,
    TrigramsAndDistanceComparator,
    TransliteratedTrigramsAndDistanceComparator,
):
    """
    Класс для нечеткого сравнения контрольных ответов, основанный на применении нескольких
    преобразований и способов сравнения. Не используется NFD. Вместо DistanceComparator используем
    TrigramsAndDistanceComparator
    """


class FuzzyStrictComparator(
    InitialComparator,
    NFKCLowerConverter,
    DistanceComparator,
):
    """
    Класс для нечеткого сравнения строк, не допускающих транслитерации, таких, как адреса
    электронной почты, телефоны.
    """
