# -*- coding: utf-8 -*-
from collections import namedtuple
import string

from passport.backend.core.compare.equality.xlit_data import (
    XLIT_EMPTY_REPS,
    XLIT_EQUIVALENTS,
    XLIT_MAX_EQ_LEN,
    XLIT_MAX_ORIG_LEN,
    XLIT_REPLACEMENTS,
)
from six.moves import xrange
from unidecode import unidecode


XlitCompareResult = namedtuple('XlitCompareResult', ['status', 'suffix_a', 'suffix_b'])


def xlit_compare(a, b):
    """
    Сравнение двух строк с учетом возможных транслитераций.
    Обе строки по возможности должны быть в нижнем регистре!
    Если свести строки не удалось, возвращаются несведенные остатки.
    @param a: первая строка
    @param b: вторая строка
    @return объект XlitCompareResult, содержащий признак равенства (status),
     остаток первой строки (suffix_a), остаток второй строки (suffix_b)
    """
    def _swap_false_result(swap, a, b):
        """
        Вспомогательная функция для возврата остатков в правильном порядке
        """
        if swap:
            return XlitCompareResult(False, b, a)
        return XlitCompareResult(False, a, b)

    swap = False
    while True:
        # пропускаем одинаковые символы, хотя это и не всегда допустимо.
        # в т.ч. если есть эквивалентные замены, начинающиеся с одинакового префикса,
        # и приводящие к одинаковому результату, мы их можем не учесть.
        a, b = _strip_common_prefix(a, b)

        if len(a) == 0 and len(b) == 0:
            return XlitCompareResult(True, '', '')

        if len(a) == 0 or len(b) == 0:
            if a in XLIT_EMPTY_REPS or b in XLIT_EMPTY_REPS:
                # одна строка пустая, другая заменяется на пустую
                return XlitCompareResult(True, '', '')
            return _swap_false_result(swap, a, b)

        # определяем строку, начинающуюся с не-ASCII символа
        if a[0] not in string.printable:
            a, b = b, a
            swap = not swap  # обновляем признак того, что строки поменялись местами

        matching_reps = _get_matching_replacements(b, a)
        if len(matching_reps) == 1:
            # быстрый случай - подходит только одна замена
            rep_from, rep_to = matching_reps[0]
            b = b.replace(rep_from, rep_to, 1)
            continue
        elif len(matching_reps) > 1:
            # медленный случай - пробуем подходящие замены рекурсивно
            results = []
            for rep_from, rep_to in matching_reps:
                res = xlit_compare(a, b.replace(rep_from, rep_to, 1))
                if res.status:
                    # в рекурсивном вызове удалось сматчить строки
                    return res
                results.append(res)
            # найдем лучшее получившееся совпадение по суммарной длине остатков
            results.sort(key=lambda x: len(x.suffix_a) + len(x.suffix_b))
            res = results[0]
            return _swap_false_result(swap, res.suffix_a, res.suffix_b)

        # не удалось ничего сматчить
        return _swap_false_result(swap, a, b)


def xlit_basic(a):
    """
    Транслитерация строки с использованием замен unidecode
    """
    return unidecode(a).lower()


def _get_prefixes(s, max_prefix_len):
    """
    Генератор для получения префиксов строки длины не более max_prefix_len,
    начиная с самого длинного
    """
    for i in xrange(min(len(s), max_prefix_len), 0, -1):
        yield s[0:i]


def _strip_common_prefix(a, b):
    """
    Удаление наиболее длинного общего префикса
    """
    ln = min(len(a), len(b))
    for i in xrange(ln):
        if a[i] != b[i]:
            return a[i:], b[i:]
    return a[ln:], b[ln:]


def _get_matching_replacements(orig_string, target_string):
    """
    Получение списка подходящих замен для начала строки orig_string и первого
    символа строки target_string. Сначала проверяется наличие замен, определенных в
    файле xlit_data, если замен не найдено, обращаемся в unidecode.
    @param orig_string: строка, к началу которой нужно применить замену
    @param target_string: строка, к которой нужно привести orig_string путем замены
    @return список подходящих замен - кортежей вида ('ф', 'f')
    """
    target_reps = []
    for replacements, max_prefix_len in ((XLIT_REPLACEMENTS, XLIT_MAX_ORIG_LEN), (XLIT_EQUIVALENTS, XLIT_MAX_EQ_LEN)):
        for prefix in _get_prefixes(orig_string, max_prefix_len):
            rep_to_list = replacements.get(prefix, [])
            for rep_to in rep_to_list:
                if rep_to != '' and rep_to[0] != target_string[0]:
                    # замена не подходит, первый получаемый символ не совпадает
                    # с первым символом целевой строки
                    continue
                target_reps.append((prefix, rep_to))

    if not target_reps:
        # в наших заменах ничего не нашлось, пробуем unidecode
        # важно: часть полученной строки может быть в верхнем регистре, например, в случае иероглифов
        translit = unidecode(orig_string[0]).lower()
        if translit and translit[0] == target_string[0]:
            # базовая замена из unidecode подходит
            target_reps = [(orig_string[0], translit)]
    return target_reps
