# -*- coding: utf-8 -*-
import logging

from six import (
    iteritems,
    iterkeys,
    itervalues,
)


log = logging.getLogger('passport.compare.equality.xlit_helper')


def build_replacements(raw_reps):
    """
    Вспомогательная функция для построения набора замен в виде, удобном для
    работы алгоритма сравнения
    @param raw_reps: замены, заданные в виде словаря {'lang': (('ф', 'f'), ('в', 'v'))}
    @return кортеж (набор замен, максимальная длина заменяемой подстроки в наборе,
     множество замен на пустую строку)
    """
    replacements = {}
    empty_replacements = set()
    for lang, lang_reps in iteritems(raw_reps):
        for rep_from, rep_to in lang_reps:
            if rep_from == rep_to:
                raise ValueError(u'Infinite replacement %s -> %s for language %s' % (rep_from, rep_to, lang))
            replacements.setdefault(rep_from, [])
            replacements[rep_from].append(rep_to)
            if rep_to == '':
                # пустая замена
                empty_replacements.add(rep_from)

    for rep_to_list in itervalues(replacements):
        # отсортируем замены в обратном порядке - сначала предлагаем самые длинные
        rep_to_list.sort(reverse=True)
    max_len = max([len(k) for k in iterkeys(replacements)])
    return replacements, max_len, empty_replacements


def build_equivalents(processed_reps):
    """
    Вспомогательная функция для построения набора замен ASCII-ASCII на основе
    заданного набора замен. Например, на основе замен 'в'-'v' и 'в'-'w' строятся
    замены 'v'-'w', 'w'-'v'.
    @param processed_reps: замены в виде словаря {'from': ['variant1', 'variant2', ...]}
    @return кортеж (набор замен ASCII-ASCII, максимальная длина заменяемой подстроки в наборе)
    """
    equivalents = {}
    for rep_from, rep_to_list in iteritems(processed_reps):
        rep_to_list = [x for x in rep_to_list if x != '']
        if len(rep_to_list) > 1:
            # замены возникают при наличии более одного варианта преобразования
            # здесь решается задача union-find, неэффективно, но один раз при
            # инициализации модуля
            eq_class = set(rep_to_list)
            orig_eq_class = set()
            while orig_eq_class != eq_class:
                # цикл продолжается пока в множество эквивалентных замен есть что добавить
                orig_eq_class = eq_class.copy()
                for rep_to in orig_eq_class:
                    eq_class.update(equivalents.get(rep_to, set()))
            for rep_to in eq_class:
                equivalents[rep_to] = eq_class
    for rep_from, to_set in iteritems(equivalents):
        # преобразуем множества в отсортированный список замен
        rep_to_list = list(to_set)
        rep_to_list.remove(rep_from)
        equivalents[rep_from] = sorted(rep_to_list, reverse=True)
    max_len = max([len(k) for k in iterkeys(equivalents)])
    return equivalents, max_len
