# -*- coding: utf-8 -*-
import logging
import time

from passport.backend.core.conf import default_settings
from passport.backend.core.dynamic_config import BaseDynamicPyModule


log = logging.getLogger('passport.conf')


class DynamicSettings(BaseDynamicPyModule):
    # Динамические сеттинги должны подновляться при перезапуске приложения, а также перед началом обработки запроса.
    # Если они не подновляются более MAX_DELAY секунд - повод зажигать мониторинг.
    # Значение стоит выбирать таким, чтобы приложение успевало за это время рестартнуть.
    MAX_DELAY = 300

    def __getattr__(self, name):
        delay = time.time() - self.expires_at
        if delay > self.MAX_DELAY:
            log.error('Dynamic settings `%s` are outdated for %s seconds', self.import_as_name, int(delay))
        try:
            return getattr(self.config, name)
        except AttributeError:
            return super(DynamicSettings, self).__getattr__(name)


class Settings(object):
    def __init__(self):
        self._settings = None
        self._dynamic_settings = None
        self._options = dict()

    def __setattr__(self, name, value):
        if name in ['_dynamic_settings', '_options', '_settings']:
            self.__dict__[name] = value
        else:
            self._options[name] = value

    def __getattr__(self, name):
        if name in self._options:
            return self._options[name]
        try:
            return getattr(self._dynamic_settings, name)
        except AttributeError:
            try:
                return getattr(self._settings, name)
            except AttributeError:
                return getattr(default_settings, name)

    def __dir__(self):
        return list(self._options.keys()) + dir(self._settings)

    def configure(self, settings=None, dynamic_settings=None, **options):
        if self.configured:
            raise RuntimeError('Settings already configured.')
        if isinstance(settings, dict):
            raise ValueError('Unable to use dict as `settings`. Pass it as `options` instead')
        if dynamic_settings and not isinstance(dynamic_settings, DynamicSettings):
            raise ValueError('`dynamic_settings` must be an instance of DynamicSettings')

        self._settings = settings
        self._dynamic_settings = dynamic_settings
        self._options.update(**options)

        self.try_reload_dynamic_settings()

    def try_reload_dynamic_settings(self):
        if self._dynamic_settings:
            self._dynamic_settings.load()

    @property
    def configured(self):
        return bool(self._settings or self._dynamic_settings or self._options)


settings = Settings()
